/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.build.testConfiguration

import com.android.build.api.variant.BuiltArtifactsLoader
import java.io.File
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity
import org.gradle.api.tasks.SkipWhenEmpty
import org.gradle.api.tasks.TaskAction
import org.gradle.work.DisableCachingByDefault

/**
 * Copy test APK (androidTest or standalone test) needed for building androidTest.zip
 *
 * If test requires instrumented app apks, they will be copied separately in either
 * [CopyApkFromArtifactsTask] or [CopyApksFromOutputProviderTask] depending on project type.
 */
@DisableCachingByDefault(because = "Only filesystem operations")
abstract class CopyTestApksTask : DefaultTask() {

    /** File existence check to determine whether to run this task. */
    @get:InputFiles
    @get:SkipWhenEmpty
    @get:PathSensitive(PathSensitivity.NONE)
    abstract val androidTestSourceCode: ConfigurableFileCollection

    @get:InputFiles
    @get:PathSensitive(PathSensitivity.NONE)
    abstract val testFolder: DirectoryProperty

    @get:Internal abstract val testLoader: Property<BuiltArtifactsLoader>

    @get:OutputFile abstract val outputApplicationId: RegularFileProperty

    @get:OutputFile abstract val outputTestApk: RegularFileProperty

    @TaskAction
    fun createApks() {
        val testApk =
            testLoader.get().load(testFolder.get())
                ?: throw RuntimeException("Cannot load required APK for task: $name")
        val testApkBuiltArtifact = testApk.elements.single()
        val destinationApk = outputTestApk.get().asFile
        File(testApkBuiltArtifact.outputFile).copyTo(destinationApk, overwrite = true)

        val outputApplicationIdFile = outputApplicationId.get().asFile
        outputApplicationIdFile.writeText(testApk.applicationId)
    }
}
