/*
 * Copyright 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.emoji2.emojipicker

import android.content.Context
import androidx.core.content.res.use
import androidx.emoji2.emojipicker.utils.FileCache
import androidx.emoji2.emojipicker.utils.UnicodeRenderableManager

/**
 * A data loader that loads the following objects either from file based caches or from resources.
 *
 * @property categorizedEmojiData: a list that holds bundled emoji separated by category, filtered
 * by renderability check. This is the data source for EmojiPickerView.
 *
 * @property emojiVariantsLookup: a map of emoji variants in bundled emoji, keyed by the primary
 * emoji. This allows faster variants lookup.
 */
internal object BundledEmojiListLoader {
    private var _categorizedEmojiData: List<EmojiDataCategory>? = null
    private var _emojiVariantsLookup: Map<String, List<String>>? = null

    internal fun load(context: Context, emojiCompatMetadata: EmojiPickerView.EmojiCompatMetadata) {
        val categoryNames = context.resources.getStringArray(R.array.category_names)

        _categorizedEmojiData = context.resources
            .obtainTypedArray(R.array.emoji_by_category_raw_resources)
            .use { ta ->
                val emojiFileCache = FileCache.getInstance(context)
                (0 until ta.length()).map {
                    val cacheFileName = getCacheFileName(it, emojiCompatMetadata)
                    emojiFileCache.getOrPut(cacheFileName) {
                        loadSingleCategory(
                            context,
                            emojiCompatMetadata,
                            ta.getResourceId(it, 0)
                        )
                    }.let { data -> EmojiDataCategory(categoryNames[it], data) }
                }.toList()
            }

        _emojiVariantsLookup =
            _categorizedEmojiData!!
                .map { it.emojiDataList }
                .flatten()
                .filter { it.variants.isNotEmpty() }
                .associate { it.primary to it.variants }
    }

    internal val categorizedEmojiData: List<EmojiDataCategory>
        get() = _categorizedEmojiData
            ?: throw IllegalStateException("BundledEmojiListLoader.load is not called")

    internal val emojiVariantsLookup: Map<String, List<String>>
        get() = _emojiVariantsLookup
            ?: throw IllegalStateException("BundledEmojiListLoader.load is not called")

    private fun loadSingleCategory(
        context: Context,
        emojiCompatMetadata: EmojiPickerView.EmojiCompatMetadata,
        resId: Int,
    ): List<EmojiData> =
        context.resources
            .openRawResource(resId)
            .bufferedReader()
            .useLines { it.toList() }
            .map { filterRenderableEmojis(it.split(","), emojiCompatMetadata) }
            .filter { it.isNotEmpty() }
            .map { EmojiData(it.first(), it.drop(1)) }

    private fun getCacheFileName(
        categoryIndex: Int,
        emojiCompatMetadata: EmojiPickerView.EmojiCompatMetadata
    ) = StringBuilder().append("emoji.v1.")
        .append(emojiCompatMetadata.hashCode())
        .append(".")
        .append(categoryIndex)
        .append(".")
        .append(
            if (UnicodeRenderableManager.isEmoji12Supported(emojiCompatMetadata)) 1 else 0
        ).toString()

    /**
     * To eliminate 'Tofu' (the fallback glyph when an emoji is not renderable), check the
     * renderability of emojis and keep only when they are renderable on the current device.
     */
    private fun filterRenderableEmojis(
        emojiList: List<String>,
        emojiCompatMetadata: EmojiPickerView.EmojiCompatMetadata,
    ) = emojiList.filter {
        UnicodeRenderableManager.isEmojiRenderable(it, emojiCompatMetadata)
    }.toList()

    internal data class EmojiData(val primary: String, val variants: List<String>)

    internal data class EmojiDataCategory(
        val categoryName: String,
        val emojiDataList: List<EmojiData>
    )
}