/*
 * Copyright 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.camera.camera2.internal;

import static androidx.camera.camera2.internal.Camera2CameraImpl.StateCallback.CameraReopenMonitor.REOPEN_LIMIT_MS;
import static androidx.camera.camera2.internal.Camera2CameraImpl.StateCallback.REOPEN_DELAY_MS;

import static com.google.common.truth.Truth.assertThat;

import static org.junit.Assume.assumeFalse;

import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.os.Handler;
import android.os.HandlerThread;

import androidx.annotation.GuardedBy;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.camera.camera2.internal.compat.CameraAccessExceptionCompat;
import androidx.camera.camera2.internal.compat.CameraManagerCompat;
import androidx.camera.camera2.internal.compat.CameraManagerCompat.CameraManagerCompatImpl;
import androidx.camera.core.CameraSelector;
import androidx.camera.core.CameraUnavailableException;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.CameraInternal;
import androidx.camera.core.impl.CameraStateRegistry;
import androidx.camera.core.impl.Observable;
import androidx.camera.core.impl.utils.MainThreadAsyncHandler;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.testing.CameraUtil;
import androidx.core.os.HandlerCompat;
import androidx.test.core.app.ApplicationProvider;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import androidx.test.filters.LargeTest;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestRule;
import org.junit.runner.RunWith;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

/** Contains {@link Camera2CameraImpl} tests for reopening the camera with failures. */
@LargeTest
@RunWith(AndroidJUnit4.class)
public final class Camera2CameraImplCameraReopenTest {

    @Rule
    public TestRule mCameraRule = CameraUtil.grantCameraPermissionAndPreTest();

    private static HandlerThread sCameraHandlerThread;
    private static Handler sCameraHandler;
    private static ExecutorService sCameraExecutor;
    private Camera2CameraImpl mCamera2CameraImpl;

    @BeforeClass
    public static void classSetup() {
        sCameraHandlerThread = new HandlerThread("cameraThread");
        sCameraHandlerThread.start();
        sCameraHandler = HandlerCompat.createAsync(sCameraHandlerThread.getLooper());
        sCameraExecutor = CameraXExecutors.newHandlerExecutor(sCameraHandler);
    }

    @AfterClass
    public static void classTeardown() {
        sCameraHandlerThread.quitSafely();
    }

    @After
    public void testTeardown() throws InterruptedException, ExecutionException {
        // Release camera, otherwise the CameraDevice is not closed, which can cause problems that
        // interfere with other tests.
        if (mCamera2CameraImpl != null) {
            mCamera2CameraImpl.release().get();
            mCamera2CameraImpl = null;
        }
    }

    @Test
    public void openCameraAfterMultipleFailures_whenCameraReopenLimitNotReached() throws Exception {
        // Set up the camera
        final FailCameraOpenCameraManagerImpl cameraManagerImpl =
                new FailCameraOpenCameraManagerImpl();
        setUpCamera(cameraManagerImpl);

        // Set up camera open attempt listener
        final Semaphore cameraOpenSemaphore = new Semaphore(0);
        cameraManagerImpl.setOnCameraOpenAttemptListener(cameraOpenSemaphore::release);

        // Try opening the camera. This will fail and trigger reopening attempts
        mCamera2CameraImpl.open();
        assertThat(cameraOpenSemaphore.tryAcquire(1, 1_000, TimeUnit.MILLISECONDS)).isTrue();

        // Wait for half the max reopen attempts to occur
        assertThat(cameraOpenSemaphore.tryAcquire(getMaxCameraReopenAttempts() / 2, REOPEN_LIMIT_MS,
                TimeUnit.MILLISECONDS)).isTrue();

        // Allow camera opening to succeed
        cameraManagerImpl.setShouldFailCameraOpen(false);

        // Verify the camera opens
        awaitCameraOpen();
    }

    @Test
    public void doNotAttemptCameraReopen_whenCameraReopenLimitReached() throws Exception {
        // Set up the camera
        final FailCameraOpenCameraManagerImpl cameraManagerImpl =
                new FailCameraOpenCameraManagerImpl();
        setUpCamera(cameraManagerImpl);

        // Set up camera open attempt listener
        final Semaphore cameraOpenSemaphore = new Semaphore(0);
        cameraManagerImpl.setOnCameraOpenAttemptListener(cameraOpenSemaphore::release);

        // Try opening the camera. This will fail and trigger reopening attempts
        mCamera2CameraImpl.open();
        assertThat(cameraOpenSemaphore.tryAcquire(1, 1_000, TimeUnit.MILLISECONDS)).isTrue();

        // Wait for approximately max reopen attempts to occur. The exact max number of reopen
        // attempts may not occur, this is due to impreciseness of the camera reopen scheduling.
        cameraOpenSemaphore.tryAcquire(getMaxCameraReopenAttempts(), REOPEN_LIMIT_MS,
                TimeUnit.MILLISECONDS);
        cameraOpenSemaphore.drainPermits();

        // Verify 0 or at most 1 camera reopen occurred. There may be 1 additional reopen attempt
        // if a last reopen attempt was scheduled right before the reopen attempt limit time was
        // reached.
        assertThat(cameraOpenSemaphore.tryAcquire(2, 2_000, TimeUnit.MILLISECONDS)).isFalse();
    }

    @Test
    public void openCameraAfterExplicitRequest_whenCameraReopenLimitReached() throws Exception {
        // Set up the camera
        final FailCameraOpenCameraManagerImpl cameraManagerImpl =
                new FailCameraOpenCameraManagerImpl();
        setUpCamera(cameraManagerImpl);

        // Set up camera open attempt listener
        final Semaphore cameraOpenSemaphore = new Semaphore(0);
        cameraManagerImpl.setOnCameraOpenAttemptListener(cameraOpenSemaphore::release);

        // Try opening the camera. This will fail and trigger reopening attempts
        mCamera2CameraImpl.open();
        assertThat(cameraOpenSemaphore.tryAcquire(1, 1_000, TimeUnit.MILLISECONDS)).isTrue();

        // Wait for approximately max reopen attempts to occur. The exact max number of reopen
        // attempts may not occur, this is due to impreciseness of the camera reopen scheduling.
        cameraOpenSemaphore.tryAcquire(getMaxCameraReopenAttempts(), REOPEN_LIMIT_MS,
                TimeUnit.MILLISECONDS);

        // Allow camera opening to succeed
        cameraManagerImpl.setShouldFailCameraOpen(false);

        // Try opening the camera. This should succeed
        mCamera2CameraImpl.open();

        // Verify the camera opens
        awaitCameraOpen();
    }

    private void setUpCamera(@NonNull final FailCameraOpenCameraManagerImpl cameraManagerImpl)
            throws CameraAccessExceptionCompat, CameraUnavailableException {
        // Get camera id of camera to open
        final String cameraId = CameraUtil.getCameraIdWithLensFacing(
                CameraSelector.LENS_FACING_BACK);
        assumeFalse("Device doesn't have a back facing camera", cameraId == null);

        // Build camera manager wrapper
        final CameraManagerCompat cameraManagerCompat = CameraManagerCompat.from(cameraManagerImpl);

        // Build camera info
        final Camera2CameraInfoImpl camera2CameraInfo = new Camera2CameraInfoImpl(cameraId,
                cameraManagerCompat.getCameraCharacteristicsCompat(cameraId));

        // Initialize camera instance
        mCamera2CameraImpl = new Camera2CameraImpl(cameraManagerCompat, cameraId,
                camera2CameraInfo, new CameraStateRegistry(1), sCameraExecutor, sCameraHandler);
    }

    @SuppressWarnings("SameParameterValue")
    private void awaitCameraOpen() throws InterruptedException {
        final Semaphore cameraOpenSemaphore = new Semaphore(0);
        final Observable.Observer<CameraInternal.State> observer =
                new Observable.Observer<CameraInternal.State>() {
                    @Override
                    public void onNewData(@Nullable CameraInternal.State state) {
                        if (state == CameraInternal.State.OPEN) {
                            cameraOpenSemaphore.release();
                        }
                    }

                    @Override
                    public void onError(@NonNull Throwable t) {
                        Logger.e("CameraReopenTest", "Camera state error: " + t.getMessage());
                    }
                };
        mCamera2CameraImpl.getCameraState().addObserver(CameraXExecutors.directExecutor(),
                observer);

        try {
            assertThat(cameraOpenSemaphore.tryAcquire(2_000, TimeUnit.MILLISECONDS)).isTrue();
        } finally {
            mCamera2CameraImpl.getCameraState().removeObserver(observer);
        }
    }

    private int getMaxCameraReopenAttempts() {
        return (int) Math.ceil(REOPEN_LIMIT_MS / (double) REOPEN_DELAY_MS);
    }

    /**
     * Wraps a {@link CameraManagerCompatImpl} instance and controls camera opening by either
     * allowing it to succeed or fail.
     */
    static class FailCameraOpenCameraManagerImpl implements CameraManagerCompatImpl {

        @NonNull
        private final CameraManagerCompatImpl mForwardCameraManagerCompatImpl;
        @NonNull
        private final Object mLock = new Object();
        @GuardedBy("mLock")
        @Nullable
        private OnCameraOpenAttemptListener mOnCameraOpenAttemptListener;
        @GuardedBy("mLock")
        private boolean mShouldFailCameraOpen = true;

        FailCameraOpenCameraManagerImpl() {
            mForwardCameraManagerCompatImpl = CameraManagerCompatImpl.from(
                    ApplicationProvider.getApplicationContext(),
                    MainThreadAsyncHandler.getInstance());
        }

        @NonNull
        @Override
        public String[] getCameraIdList() throws CameraAccessExceptionCompat {
            return mForwardCameraManagerCompatImpl.getCameraIdList();
        }

        @Override
        public void registerAvailabilityCallback(@NonNull Executor executor,
                @NonNull CameraManager.AvailabilityCallback callback) {
            mForwardCameraManagerCompatImpl.registerAvailabilityCallback(executor, callback);
        }

        @Override
        public void unregisterAvailabilityCallback(
                @NonNull CameraManager.AvailabilityCallback callback) {
            mForwardCameraManagerCompatImpl.unregisterAvailabilityCallback(callback);
        }

        @NonNull
        @Override
        public CameraCharacteristics getCameraCharacteristics(@NonNull String cameraId)
                throws CameraAccessExceptionCompat {
            return mForwardCameraManagerCompatImpl.getCameraCharacteristics(cameraId);
        }

        @NonNull
        @Override
        public CameraManager getCameraManager() {
            return mForwardCameraManagerCompatImpl.getCameraManager();
        }

        @Override
        public void openCamera(@NonNull String cameraId, @NonNull Executor executor,
                @NonNull CameraDevice.StateCallback callback) throws CameraAccessExceptionCompat {
            synchronized (mLock) {
                if (mOnCameraOpenAttemptListener != null) {
                    mOnCameraOpenAttemptListener.onCameraOpenAttempt();
                }
                if (mShouldFailCameraOpen) {
                    // Throw any exception
                    throw new SecurityException("Lacking privileges");
                } else {
                    mForwardCameraManagerCompatImpl.openCamera(cameraId, executor, callback);
                }
            }
        }

        public void setShouldFailCameraOpen(boolean shouldFailCameraOpen) {
            synchronized (mLock) {
                this.mShouldFailCameraOpen = shouldFailCameraOpen;
            }
        }

        public void setOnCameraOpenAttemptListener(
                @NonNull final OnCameraOpenAttemptListener listener) {
            synchronized (mLock) {
                this.mOnCameraOpenAttemptListener = listener;
            }
        }

        interface OnCameraOpenAttemptListener {
            /** Triggered whenever an attempt to open the camera is made. */
            void onCameraOpenAttempt();
        }
    }
}
