/*
 * Copyright 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.appcompat.content.res;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.drawable.Drawable;

import androidx.annotation.ColorRes;
import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.ResourceManagerInternal;
import androidx.core.content.ContextCompat;

/**
 * Class for accessing an application's resources through AppCompat, and thus any backward
 * compatible functionality.
 */
@SuppressLint("RestrictedAPI") // Temporary until we have correct restriction scopes for 1.0
public final class AppCompatResources {

    private AppCompatResources() {}

    /**
     * Returns the {@link ColorStateList} from the given resource. The resource can include
     * themeable attributes, regardless of API level.
     *
     * @param context context to inflate against
     * @param resId the resource identifier of the ColorStateList to retrieve
     */
    public static ColorStateList getColorStateList(@NonNull Context context, @ColorRes int resId) {
        return ContextCompat.getColorStateList(context, resId);
    }

    /**
     * Return a drawable object associated with a particular resource ID.
     *
     * <p>This method supports inflation of {@code <vector>}, {@code <animated-vector>} and
     * {@code <animated-selector>} resources on devices where platform support is not available.</p>
     *
     * @param context context to inflate against
     * @param resId   The desired resource identifier, as generated by the aapt
     *                tool. This integer encodes the package, type, and resource
     *                entry. The value 0 is an invalid identifier.
     * @return Drawable An object that can be used to draw this resource.
     * @see ContextCompat#getDrawable(Context, int)
     */
    @Nullable
    public static Drawable getDrawable(@NonNull Context context, @DrawableRes int resId) {
        return ResourceManagerInternal.get().getDrawable(context, resId);
    }

}
