/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.ui.androidview.demos

import android.graphics.Color
import android.view.View
import android.view.ViewGroup
import androidx.compose.Composable
import androidx.ui.androidview.adapters.setOnClick
import androidx.ui.core.Alignment
import androidx.ui.core.Modifier
import androidx.ui.core.gesture.tapGestureFilter
import androidx.ui.demos.common.ComposableDemo
import androidx.ui.demos.common.DemoCategory
import androidx.ui.foundation.Box
import androidx.ui.foundation.HorizontalScroller
import androidx.ui.foundation.Text
import androidx.ui.foundation.VerticalScroller
import androidx.ui.foundation.drawBackground
import androidx.ui.layout.Column
import androidx.ui.layout.fillMaxSize
import androidx.ui.layout.padding
import androidx.ui.layout.preferredHeight
import androidx.ui.layout.preferredSize
import androidx.ui.layout.wrapContentSize
import androidx.ui.geometry.Offset
import androidx.ui.unit.dp
import androidx.ui.viewinterop.AndroidView

val AndroidInComposeDemos = DemoCategory("Android In Compose Interop", listOf(
    ComposableDemo("4 Android tap in Compose") { FourAndroidTapInCompose() },
    ComposableDemo("Android tap in Compose tap") { AndroidTapInComposeTap() },
    ComposableDemo("Android tap in Compose scroll") { AndroidTapInComposeScroll() },
    ComposableDemo("Android scroll in Compose scroll (different orientation)") {
        AndroidScrollInComposeScrollDifferentOrientation()
    },
    ComposableDemo("Android scroll in Compose scroll (same orientation)") {
        AndroidScrollInComposeScrollSameOrientation()
    }
))

@Composable
private fun FourAndroidTapInCompose() {
    Column {
        Text("Demonstrates that pointer locations are dispatched to Android correctly.")
        Text(
            "Below is a ViewGroup with 4 Android buttons in it.  When each button is tapped, the" +
                    " background of the ViewGroup is updated."
        )
        Box(
            Modifier
                .fillMaxSize()
                .wrapContentSize(Alignment.Center)
                .preferredSize(240.dp)
        ) {
            AndroidView(R.layout.android_4_buttons_in_compose) { view ->
                view as ViewGroup
                view.findViewById<View>(R.id.buttonBlue).setOnClick {
                    view.setBackgroundColor(Color.BLUE)
                }
                view.findViewById<View>(R.id.buttonRed).setOnClick {
                    view.setBackgroundColor(Color.RED)
                }
                view.findViewById<View>(R.id.buttonGreen).setOnClick {
                    view.setBackgroundColor(Color.GREEN)
                }
                view.findViewById<View>(R.id.buttonYellow).setOnClick {
                    view.setBackgroundColor(Color.YELLOW)
                }
            }
        }
    }
}

@Composable
private fun AndroidTapInComposeTap() {
    var theView: View? = null

    val onTap: (Offset) -> Unit = {
        theView?.setBackgroundColor(Color.BLUE)
    }

    Column {
        Text(
            "Demonstrates that pointer input interop is working correctly in the simple case of " +
                    "tapping."
        )
        Text(
            "Below there is an Android ViewGroup with a button in it.  The whole thing is wrapped" +
                    " in a Box with a tapGestureFilter modifier on it.  When you click the " +
                    "button, the ViewGroup's background turns red.  When you click anywhere else " +
                    "in the ViewGroup, the tapGestureFilter \"fires\" and the background turns " +
                    "Blue."
        )
        Box(
            Modifier
                .tapGestureFilter(onTap)
                .fillMaxSize()
                .wrapContentSize(Alignment.Center)
                .preferredSize(240.dp)
        ) {
            AndroidView(R.layout.android_tap_in_compose_tap) { view ->
                theView = view
                theView?.setBackgroundColor(Color.GREEN)
                view.findViewById<View>(R.id.buttonRed).setOnClick {
                    theView?.setBackgroundColor(Color.RED)
                }
            }
        }
    }
}

@Composable
private fun AndroidTapInComposeScroll() {
    Column {
        Text(
            "Demonstrates that pointer input interop is working correctly when tappable things in" +
                    " Android are put inside of something scrollable in Compose."
        )
        Text(
            "Below is a Compose HorizontalScroller with a wide horizontal LinearLayout in it, " +
                    "that is comprised of 4 buttons.  Clicking buttons changes the LinearLayout's" +
                    " background color.  When you drag horizontally, the HorizontalScroller drags" +
                    ". If a pointer starts on a button and then drags horizontally, the button " +
                    "will not be clicked when released."
        )
        HorizontalScroller {
            AndroidView(R.layout.android_tap_in_compose_scroll) { view ->
                view.setBackgroundColor(Color.YELLOW)
                view.findViewById<View>(R.id.buttonRed).apply {
                    isClickable = false
                    setOnClick {
                        view.setBackgroundColor(Color.RED)
                    }
                }
                view.findViewById<View>(R.id.buttonGreen).apply {
                    isClickable = false
                    setOnClick {
                        view.setBackgroundColor(Color.GREEN)
                    }
                }
                view.findViewById<View>(R.id.buttonBlue).apply {
                    isClickable = false
                    setOnClick {
                        view.setBackgroundColor(Color.BLUE)
                    }
                }
                view.findViewById<View>(R.id.buttonYellow).apply {
                    isClickable = false
                    setOnClick {
                        view.setBackgroundColor(Color.YELLOW)
                    }
                }
            }
        }
    }
}

@Composable
private fun AndroidScrollInComposeScrollDifferentOrientation() {
    Column {
        Text("Demonstrates correct \"scroll orientation\" locking when something scrollable in " +
                "Android is nested inside something scrollable in Compose.")
        Text("You should only be able to scroll in one orientation at a time.")
        HorizontalScroller(modifier = Modifier.drawBackground(androidx.ui.graphics.Color.Blue)) {
            Box(modifier = Modifier.padding(96.dp).drawBackground(androidx.ui.graphics.Color.Red)) {
                AndroidView(R.layout.android_scroll_in_compose_scroll_different_orientation)
            }
        }
    }
}

@Composable
private fun AndroidScrollInComposeScrollSameOrientation() {
    Column {
        Text("Supposed to demonstrate correct nested scrolling when something scrollable in " +
                "Android is inside something scrollable in Compose.")
        Text(
            "This doesn't actually work because nested scrolling isn't implemented between " +
                    "Compose and Android.  Normally, this lack of implementation would mean the " +
                    "parent would always intercept first and thus block the child from ever " +
                    "scrolling. However, currently, the touch slop for Android is smaller than " +
                    "that for Compose, and thus the child scrolls and prevents the parent from " +
                    "intercepting. "
        )
        VerticalScroller(modifier = Modifier.drawBackground(androidx.ui.graphics.Color.Blue)) {
            Box(
                modifier = Modifier
                    .padding(96.dp)
                    .drawBackground(androidx.ui.graphics.Color.Red)
                    .preferredHeight(750.dp)
            ) {
                AndroidView(R.layout.android_scroll_in_compose_scroll_same_orientation)
            }
        }
    }
}