/*
 * Copyright 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.build.dependencyallowlist

import org.junit.Assert.assertEquals
import org.junit.Test

class DependencyAllowlistTest {
    @Test
    fun allComponentsHaveBugs() {
        val verificationMetadataXml = """
          <?xml version="1.0" encoding="UTF-8"?>
          <verification-metadata xmlns="https://schema.gradle.org/dependency-verification" 
                                 xmlns:androidx="https://developer.android.com/jetpack/androidx">
            <components>
               <component group="g" name="g" version="3.1" androidx:reason="Unsigned b/8675309">
                 <artifact name="g-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>
            </components>
          </verification-metadata>
        """.trimIndent()
        assertEquals(listOf<String>(), allowlistWarnings(verificationMetadataXml))
    }

    @Test
    fun commentsDontCountAsReasons() {
        val verificationMetadataXml = """
          <?xml version="1.0" encoding="UTF-8"?>
          <verification-metadata xmlns="https://schema.gradle.org/dependency-verification" 
                                 xmlns:androidx="https://developer.android.com/jetpack/androidx">
            <components>
               <!-- Unsigned b/8675309 -->
               <component group="g" name="g" version="3.1">
                 <artifact name="g-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>
            </components>
          </verification-metadata>
        """.trimIndent()
        assertEquals(
            listOf(
                "Add androidx:reason for unsigned component 'g' (See go/androidx-unsigned-bugs)"
            ),
            allowlistWarnings(verificationMetadataXml)
        )
    }

    @Test
    fun oneComponentHasGitHubBug() {
        val verificationMetadataXml = """
          <?xml version="1.0" encoding="UTF-8"?>
          <verification-metadata xmlns="https://schema.gradle.org/dependency-verification" 
                                 xmlns:androidx="https://developer.android.com/jetpack/androidx">
            <components>
               <component group="g" name="g" version="3.1" 
                          androidx:reason="Unsigned https://github.com/rerepo/somecode/issues/65">
                 <artifact name="g-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>

               <component group="ng" name="g" version="3.1" 
                          androidx:reason="Not a good bug comment">
                 <artifact name="g-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>
            </components>
          </verification-metadata>
        """.trimIndent()
        assertEquals(
            listOf(
                "Add androidx:reason for unsigned component 'ng' (See go/androidx-unsigned-bugs)"
            ),
            allowlistWarnings(verificationMetadataXml)
        )
    }

    @Test
    fun oneComponentNoReason() {
        val verificationMetadataXml = """
          <?xml version="1.0" encoding="UTF-8"?>
          <verification-metadata xmlns="https://schema.gradle.org/dependency-verification" 
                                 xmlns:androidx="https://developer.android.com/jetpack/androidx">
            <components>
               <component group="g" name="g" version="3.1">
                 <artifact name="g-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>
            </components>
          </verification-metadata>
        """.trimIndent()
        assertEquals(
            listOf(
                "Add androidx:reason for unsigned component 'g' (See go/androidx-unsigned-bugs)"
            ),
            allowlistWarnings(verificationMetadataXml)
        )
    }

    @Test
    fun reasonThatIsNotABug() {
        val verificationMetadataXml = """
          <?xml version="1.0" encoding="UTF-8"?>
          <verification-metadata xmlns="https://schema.gradle.org/dependency-verification" 
                                 xmlns:androidx="https://developer.android.com/jetpack/androidx">
            <components>
               <component group="ggg" name="ggg" version="3.1" 
                          androidx:reason="This does not reference a bug">
                 <artifact name="ggg-3.1.jar">
                   <sha256
                      value="f5759b7fcdfc83a525a036deedcbd32e5b536b625ebc282426f16ca137eb5902"
                      origin="Generated by Gradle"
                   />
                 </artifact>
               </component>
            </components>
          </verification-metadata>
        """.trimIndent()
        assertEquals(
            listOf(
                "Add androidx:reason for unsigned component 'ggg' (See go/androidx-unsigned-bugs)"
            ),
            allowlistWarnings(verificationMetadataXml)
        )
    }
}
