/*
 * Copyright 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.graphics.vector

import kotlin.math.abs
import kotlin.test.Test

class FastFloatParserTest {
    @Test
    fun parseFloats() {
        val parser = PathParser()

        FloatData.forEach { line ->
            val (hex, number) = line.split(' ')
            val bits = hex.toInt(16)

            val nodes = parser.parsePathString("H$number").toNodes()

            assert(nodes.isNotEmpty()) { line }

            val x = (nodes[0] as PathNode.HorizontalTo).x

            assert(abs(x.toBits() - bits) < 2) {
                "Expected: 0x$bits\n" +
                "Actual:   0x${x.toBits()}\n" +
                "    in $line (toFloat() = ${number.toFloat()})"
            }
        }
    }
}

// Data from https://github.com/fastfloat/supplemental_test_files
// Under Apache 2.0 license
private val FloatData = arrayOf(
    "00000000 .0",
    "00000000 0",
    "00000000 0.0",
    "00000000 0.0000",
    "00000000 0e1",
    "00000000 0e2",
    "00000000 0e3",
    "00000000 0e4395",
    "00000000 0e47",
    "00000000 0e4851",
    "00000000 0e5",
    "00000000 0e7",
    "00000000 0e785",
    "00000000 0e93",
    "00000000 0e9999999999999999999999999999",
    "00000000 1e-324",
    "00000000 1e-500",
    "00000000 3e-324",
    "00000000 4.9406564584124653e-324",
    "00000000 4.9406564584124654e-324",
    "00000000 1.00000000000000188558920870223463870174566020691753515394643550663070558368373221" +
        "972569761144603605635692374830246134201063722058e-309",
    "00000000 0.00000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000002225073858507200889024586876085859" +
        "887650423112240959465493524802562440009228235695178775888803759155264230978095043431208" +
        "587738715835729182199302029437922422355981982750124204178896957131179108226104397197960" +
        "400045489739193807919893608152561311337614984204327175103362739154978273159414382813627" +
        "511383860409424946494228631669542910508020181592664213499660651780309507591305871984642" +
        "390606863710200510872328278467884363194451586613504122347901479236958520832159762106637" +
        "540161373658304419360371477835530668283453563400507407304013560296804637591858316312422" +
        "452159926254649430083685186171942241764645513713542013221703137049658321015465406803539" +
        "741790602258950302350193751977303094576317321085250729930508976158251915",
    "00000000 2.2250738585072009e-308",
    "00000000 2.2250738585072013e-308",
    "00000000 2.2250738585072014e-308",
    "00000000 0.00000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000" +
        "000000000000000000000000000000000000000000000000000004450147717014402272114819593418263" +
        "951869639092703291296046852219449644444042153891033059047816270175828298317826079242213" +
        "740172877389189291055314414815641243486759976282126534658507104573762744298025962244902" +
        "903779698114444614570510266311510031828794952795966823603998647925096578034214163701381" +
        "261333311989876551545144031526125381326665295130600018491776632866075559583739224098994" +
        "780755659409810102161219881460525874257917900007167599934414508608720568157791543592301" +
        "891033496486942061405218289243144579760516365090360651414037721744226256159024466852576" +
        "737244643007551333245007965068671949137768847800530996396770975896584413789443379662199" +
        "396731693628045708486661320679701772891608002069867940855134372886767540",
    "00000000 1.2e-307",
    "00000000 1e-300",
    "00000000 1e-256",
    "00000000 9007199254740992.e-256",
    "00000000 1e-128",
    "00000000 42823146028335318693e-128",
    "00000000 1e-64",
    "00000000 7.0060e-46",
    "00000001 7.0064923216240854e-46",
    "00000001 1.1754943508e-45",
    "00000001 1.4012984643e-45",
    "00000001 0.00000000000000000000000000000000000000000000140129846432481707092372958328991613" +
        "128026194187651577175706828388979108268586060148663818836212158203125",
    "007FFFFF 0.00000000000000000000000000000000000001175494210692441075487029444849287348827052" +
        "428745893333857174530571588870475618904265502351336181163787841796875",
    "007FFFFF 1.1754942107e-38",
    "00800000 1.1754943508e-38",
    "00800000 0.00000000000000000000000000000000000001175494350822287507968736537222245677818665" +
        "5567720875215087517062784172594547271728515625",
    "00800003 1.1754947011469036e-38",
    "00FFFFFF 0.00000000000000000000000000000000000002350988561514728583455765982071533026645717" +
        "985517980855365926236850006129930346077117064851336181163787841796875",
    "01000000 2.3509887016445750159374730744444913556373311135441750430175034126e-38",
    "01800000 4.7019774032891500318749461488889827112746622270883500860350068251e-38",
    "0A4FB11F 1e-32",
    "24E69595 1e-16",
    "29A2212D 7.2e-14",
    "29E12E13 1e-13",
    "39102534 0.00013746770127909258",
    "39BECE41 0.00036393293703440577",
    "3A6181A6 0.0008602388261351734",
    "3AD0BAE5 0.0015924838953651488",
    "3B0D2710 0.002153817447833717",
    "3B8A536D 0.004221370676532388",
    "3C467C71 0.012114629615098238",
    "3CE5EF9A 0.028068351559340954",
    "3D17CCF0 0.03706067614257336",
    "3DBE3F17 0.09289376810193062",
    "3DCCCCCD 0.1",
    "3E200000 0.15625",
    "3E200000 0.156250000000000000000000000000000000000000",
    "3E345144 0.176091259055681",
    "3E345144 0.1760912590558",
    "3E5F23F5 0.21791061013936996",
    "3E943D3B 0.289529654602168",
    "3E9A209B 0.301029995663981",
    "3E9A209B 0.30103",
    "3E9C529D 0.30531780421733856",
    "3EFFFFFD .4999999",
    "3F000000 .5",
    "3F000000 0.5",
    "3F000002 .5000001",
    "3F4322D6 0.7622503340244293",
    "3F800000 1",
    "3F800000 1e0",
    "3F800032 1.000006",
    "3F8147AE 1.01",
    "3F91EB85 1.14",
    "3F98089F 1.1877630352973938",
    "3FA68B6E 1.30113",
    "3FB0A3D7 1.38",
    "3FB33333 1.4",
    "3FC00000 1.5",
    "3FD9999A 1.7",
    "3FE66666 1.8",
    "40000000 2",
    "40000000 2.0",
    "40000000 2e0",
    "400646F1 2.09808",
    "4019999A 2.4",
    "402BFB5E 2.687217116355896",
    "402DF854 2.718281828459045",
    "402DF854 2.71828182845904523536028747135266249775724709369995",
    "40400000 3",
    "40470A3D 3.11",
    "40490FDB 3.141592653589793",
    "40490FDB 3.14159265358979323846264338327950288419716939937510",
    "40490FDB 3.14159265358979323846264338327950288419716939937510582097494459230781640628620899" +
        "86280348253421170679",
    "40490FDB 3.14159265359",
    "40490FF9 3.1416",
    "404CCCCD 3.2",
    "40733333 3.8",
    "4079999A 3.9",
    "40800000 4",
    "4083D70A 4.12",
    "40A00000 5",
    "40C00000 6",
    "40C00000 6e0",
    "40E00000 7",
    "40E00000 7e0",
    "40F17C87 7.5464513301849365",
    "41000000 8",
    "41100000 9",
    "41200000 10",
    "41200000 1e0000001",
    "41200000 1e01",
    "41200000 1e1",
    "41300000 11",
    "41400000 12",
    "41500000 13",
    "415EADE2 13.91745138168335",
    "41600000 14",
    "41607AE1 14.03",
    "41700000 15",
    "41800000 16",
    "41880000 17",
    "418BE43C 17.486443519592285",
    "41900000 18",
    "419051EC 18.04",
    "41980000 19",
    "41A00000 20",
    "41A051EC 20.04",
    "41A80000 21",
    "41B00000 22",
    "41B80000 23",
    "41C00000 24",
    "41C00000 24e0",
    "41C6E148 24.86",
    "41C80000 25",
    "41D00000 26",
    "41D00000 26e0",
    "41D80000 27",
    "41E00000 28",
    "41E80000 29",
    "41F00000 30",
    "41F00000 3e01",
    "41F00000 3e1",
    "41F80000 31",
    "42000000 32",
    "42040000 33",
    "42080000 34",
    "420C0000 35",
    "42100000 36",
    "42140000 37",
    "42180000 38",
    "421C0000 39",
    "42200000 40",
    "42200000 4e1",
    "42240000 41",
    "42280000 42",
    "422C0000 43",
    "42300000 44",
    "42340000 45",
    "42380000 46",
    "423C0000 47",
    "42400000 48",
    "42440000 49",
    "42480000 50",
    "424A70A4 50.61",
    "424B3F0E 50.811574935913086",
    "424C0000 51",
    "42500000 52",
    "42540000 53",
    "42580000 54",
    "425C0000 55",
    "42600000 56",
    "42640000 57",
    "42680000 58",
    "426C0000 59",
    "42700000 60",
    "42700000 6e1",
    "42740000 61",
    "42780000 62",
    "427C0000 63",
    "42800000 64",
    "42820000 65",
    "42840000 66",
    "42860000 67",
    "42880000 68",
    "42887333 68.225",
    "428A0000 69",
    "428C0000 70",
    "428E0000 71",
    "42900000 72",
    "42920000 73",
    "42940000 74",
    "42960000 75",
    "42980000 76",
    "429A0000 77",
    "429C0000 78",
    "429E0000 79",
    "429E0000 79e0",
    "42A00000 80",
    "42A00000 8e1",
    "42A20000 81",
    "42A40000 82",
    "42A60000 83",
    "42A80000 84",
    "42A80000 84e0",
    "42AA0000 85",
    "42AC0000 86",
    "42AD0F5C 86.53",
    "42AE0000 87",
    "42B00000 88",
    "42B20000 89",
    "42B40000 90",
    "42B60000 91",
    "42B80000 92",
    "42B80000 92e0",
    "42BA0000 93",
    "42BC0000 94",
    "42BC0000 94e0",
    "42BE0000 95",
    "42C00000 96",
    "42C20000 97",
    "42C40000 98",
    "42C60000 99",
    "42C80000 1.e2",
    "42C80000 100",
    "42C80000 10e1",
    "42C80000 1e2",
    "42CA0000 101",
    "42CC0000 102",
    "42CE0000 103",
    "42D00000 104",
    "42D20000 105",
    "42D40000 106",
    "42D60000 107",
    "42D80000 108",
    "42DA0000 109",
    "42DC0000 110",
    "42DE0000 111",
    "42E00000 112",
    "42E20000 113",
    "42E40000 114",
    "42E60000 115",
    "42E80000 116",
    "42EA0000 117",
    "42EC0000 118",
    "42EE0000 119",
    "42F00000 120",
    "42F20000 121",
    "42F40000 122",
    "42F60000 123",
    "42F80000 124",
    "42F8566C 124.16878890991211",
    "42FA0000 125",
    "42FC0000 126",
    "42FE0000 127",
    "43000000 128",
    "43010000 129",
    "43020000 130",
    "43030000 131",
    "43040000 132",
    "43050000 133",
    "43060000 134",
    "43070000 135",
    "43080000 136",
    "43090000 137",
    "430A0000 138",
    "430B0000 139",
    "430C0000 140",
    "430D0000 141",
    "430E0000 142",
    "430F0000 143",
    "43100000 144",
    "43110000 145",
    "43120000 146",
    "43130000 147",
    "43140000 148",
    "43150000 149",
    "43160000 150",
    "43170000 151",
    "43180000 152",
    "43190000 153",
    "431A0000 154",
    "431B0000 155",
    "431C0000 156",
    "431D0000 157",
    "431E0000 158",
    "431F0000 159",
    "43200000 160",
    "43210000 161",
    "43220000 162",
    "43230000 163",
    "43240000 164",
    "43250000 165",
    "43260000 166",
    "43270000 167",
    "43280000 168",
    "43290000 169",
    "432A0000 170",
    "432B0000 171",
    "432C0000 172",
    "432D0000 173",
    "432E0000 174",
    "432F0000 175",
    "43300000 176",
    "43310000 177",
    "43320000 178",
    "43330000 179",
    "43340000 180",
    "43350000 181",
    "43360000 182",
    "43370000 183",
    "43380000 184",
    "43390000 185",
    "433A0000 186",
    "433B0000 187",
    "433C0000 188",
    "433D0000 189",
    "433E0000 190",
    "433F0000 191",
    "43400000 192",
    "43410000 193",
    "43420000 194",
    "43430000 195",
    "43440000 196",
    "43450000 197",
    "43460000 198",
    "43470000 199",
    "43480000 200",
    "43490000 201",
    "434A0000 202",
    "434B0000 203",
    "434C0000 204",
    "434D0000 205",
    "434E0000 206",
    "434E80CC 206.50310516357422",
    "434F0000 207",
    "43500000 208",
    "43510000 209",
    "43520000 210",
    "43530000 211",
    "43540000 212",
    "43550000 213",
    "43560000 214",
    "43570000 215",
    "43580000 216",
    "43590000 217",
    "435A0000 218",
    "435B0000 219",
    "435C0000 220",
    "435D0000 221",
    "435E0000 222",
    "435F0000 223",
    "43600000 224",
    "43610000 225",
    "43620000 226",
    "43630000 227",
    "43640000 228",
    "43650000 229",
    "43660000 230",
    "43670000 231",
    "43680000 232",
    "43690000 233",
    "436A0000 234",
    "436B0000 235",
    "436C0000 236",
    "436D0000 237",
    "436E0000 238",
    "436F0000 239",
    "43700000 240",
    "43710000 241",
    "43720000 242",
    "43730000 243",
    "43740000 244",
    "43750000 245",
    "43760000 246",
    "43770000 247",
    "43780000 248",
    "43790000 249",
    "437A0000 250",
    "437B0000 251",
    "437C0000 252",
    "437D0000 253",
    "437E0000 254",
    "437F0000 255",
    "43800000 256",
    "43808000 257",
    "43810000 258",
    "43818000 259",
    "43820000 260",
    "43828000 261",
    "43830000 262",
    "43838000 263",
    "43840000 264",
    "43848000 265",
    "43850000 266",
    "43858000 267",
    "43860000 268",
    "43868000 269",
    "43870000 270",
    "43878000 271",
    "43880000 272",
    "43888000 273",
    "43890000 274",
    "43898000 275",
    "438A0000 276",
    "438A8000 277",
    "438B0000 278",
    "438B8000 279",
    "438C0000 280",
    "438C8000 281",
    "438D0000 282",
    "438D8000 283",
    "438E0000 284",
    "438E8000 285",
    "438F0000 286",
    "438F8000 287",
    "43900000 288",
    "43908000 289",
    "43910000 290",
    "43918000 291",
    "43920000 292",
    "43928000 293",
    "43930000 294",
    "43932A3D 294.33",
    "43938000 295",
    "43940000 296",
    "43948000 297",
    "43950000 298",
    "43958000 299",
    "43960000 300",
    "43960000 3e2",
    "43968000 301",
    "43970000 302",
    "43978000 303",
    "43980000 304",
    "43988000 305",
    "43990000 306",
    "43998000 307",
    "439A0000 308",
    "439A8000 309",
    "439B0000 310",
    "439B8000 311",
    "439C0000 312",
    "439C8000 313",
    "439D0000 314",
    "439D8000 315",
    "439E0000 316",
    "439E8000 317",
    "439F0000 318",
    "439F8000 319",
    "43A00000 320",
    "43A00000 32e1",
    "43A08000 321",
    "43A10000 322",
    "43A18000 323",
    "43A20000 324",
    "43A28000 325",
    "43A30000 326",
    "43A38000 327",
    "43A40000 328",
    "43A48000 329",
    "43A50000 330",
    "43A58000 331",
    "43A60000 332",
    "43A68000 333",
    "43A70000 334",
    "43A78000 335",
    "43A80000 336",
    "43A88000 337",
    "43A90000 338",
    "43A98000 339",
    "43AA0000 340",
    "43AA8000 341",
    "43AB0000 342",
    "43AB8000 343",
    "43AC0000 344",
    "43AC8000 345",
    "43AD0000 346",
    "43AD8000 347",
    "43AE0000 348",
    "43AE8000 349",
    "43AF0000 350",
    "43AF8000 351",
    "43B00000 352",
    "43B08000 353",
    "43B10000 354",
    "43B18000 355",
    "43B20000 356",
    "43B28000 357",
    "43B30000 358",
    "43B38000 359",
    "43B40000 360",
    "43B48000 361",
    "43B50000 362",
    "43B58000 363",
    "43B60000 364",
    "43B68000 365",
    "43B70000 366",
    "43B78000 367",
    "43B80000 368",
    "43B88000 369",
    "43B90000 370",
    "43B98000 371",
    "43BA0000 372",
    "43BA8000 373",
    "43BB0000 374",
    "43BB8000 375",
    "43BC0000 376",
    "43BC8000 377",
    "43BD0000 378",
    "43BD8000 379",
    "43BE0000 380",
    "43BE8000 381",
    "43BF0000 382",
    "43BF8000 383",
    "43C00000 384",
    "43C08000 385",
    "43C10000 386",
    "43C18000 387",
    "43C20000 388",
    "43C28000 389",
    "43C30000 390",
    "43C38000 391",
    "43C40000 392",
    "43C48000 393",
    "43C50000 394",
    "43C58000 395",
    "43C60000 396",
    "43C68000 397",
    "43C70000 398",
    "43C78000 399",
    "43C80000 400",
    "43C88000 401",
    "43C90000 402",
    "43C98000 403",
    "43CA0000 404",
    "43CA8000 405",
    "43CB0000 406",
    "43CB8000 407",
    "43CC0000 408",
    "43CC8000 409",
    "43CD0000 410",
    "43CD8000 411",
    "43CDF184 411.88682556152344",
    "43CE0000 412",
    "43CE8000 413",
    "43CF0000 414",
    "43CF8000 415",
    "43D00000 416",
    "43D08000 417",
    "43D10000 418",
    "43D18000 419",
    "43D20000 420",
    "43D28000 421",
    "43D30000 422",
    "43D38000 423",
    "43D40000 424",
    "43D48000 425",
    "43D50000 426",
    "43D58000 427",
    "43D60000 428",
    "43D68000 429",
    "43D70000 430",
    "43D78000 431",
    "43D80000 432",
    "43D88000 433",
    "43D90000 434",
    "43D98000 435",
    "43DA0000 436",
    "43DA8000 437",
    "43DB0000 438",
    "43DB8000 439",
    "43DC0000 440",
    "43DC8000 441",
    "43DD0000 442",
    "43DD8000 443",
    "43DE0000 444",
    "43DE8000 445",
    "43DF0000 446",
    "43DF8000 447",
    "43E00000 448",
    "43E08000 449",
    "43E10000 450",
    "43E18000 451",
    "43E20000 452",
    "43E28000 453",
    "43E30000 454",
    "43E38000 455",
    "43E40000 456",
    "43E48000 457",
    "43E50000 458",
    "43E58000 459",
    "43E60000 460",
    "43E68000 461",
    "43E70000 462",
    "43E78000 463",
    "43E80000 464",
    "43E88000 465",
    "43E90000 466",
    "43E98000 467",
    "43EA0000 468",
    "43EA8000 469",
    "43EB0000 470",
    "43EB8000 471",
    "43EC0000 472",
    "43EC8000 473",
    "43ED0000 474",
    "43ED8000 475",
    "43EE0000 476",
    "43EE8000 477",
    "43EF0000 478",
    "43EF8000 479",
    "43F00000 480",
    "43F08000 481",
    "43F10000 482",
    "43F18000 483",
    "43F20000 484",
    "43F28000 485",
    "43F30000 486",
    "43F38000 487",
    "43F40000 488",
    "43F48000 489",
    "43F50000 490",
    "43F58000 491",
    "43F60000 492",
    "43F68000 493",
    "43F70000 494",
    "43F78000 495",
    "43F80000 496",
    "43F88000 497",
    "43F90000 498",
    "43F98000 499",
    "43FA0000 500",
    "43FA8000 501",
    "43FB0000 502",
    "43FB8000 503",
    "43FC0000 504",
    "43FC8000 505",
    "43FD0000 506",
    "43FD8000 507",
    "43FE0000 508",
    "43FE8000 509",
    "43FF0000 510",
    "43FF8000 511",
    "44000000 512",
    "44004000 513",
    "44008000 514",
    "4400C000 515",
    "44010000 516",
    "44014000 517",
    "44018000 518",
    "4401C000 519",
    "44020000 520",
    "44024000 521",
    "44026A3D 521.66",
    "44028000 522",
    "4402C000 523",
    "44030000 524",
    "44034000 525",
    "44038000 526",
    "4403C000 527",
    "44040000 528",
    "44044000 529",
    "44048000 530",
    "4404C000 531",
    "44050000 532",
    "44054000 533",
    "44058000 534",
    "4405C000 535",
    "44060000 536",
    "44064000 537",
    "44068000 538",
    "4406C000 539",
    "44070000 540",
    "44074000 541",
    "44078000 542",
    "4407C000 543",
    "44080000 544",
    "44084000 545",
    "44088000 546",
    "4408C000 547",
    "44090000 548",
    "44094000 549",
    "44098000 550",
    "4409C000 551",
    "440A0000 552",
    "440A4000 553",
    "440A8000 554",
    "440AC000 555",
    "440B0000 556",
    "440B4000 557",
    "440B8000 558",
    "440BC000 559",
    "440C0000 560",
    "440C4000 561",
    "440C8000 562",
    "440CC000 563",
    "440D0000 564",
    "440D4000 565",
    "440D8000 566",
    "440DC000 567",
    "440E0000 568",
    "440E4000 569",
    "440E8000 570",
    "440EC000 571",
    "440F0000 572",
    "440F4000 573",
    "440F8000 574",
    "440FC000 575",
    "44100000 576",
    "44104000 577",
    "44108000 578",
    "4410C000 579",
    "44110000 580",
    "44114000 581",
    "44118000 582",
    "4411C000 583",
    "44120000 584",
    "44124000 585",
    "44128000 586",
    "4412C000 587",
    "44130000 588",
    "44134000 589",
    "44138000 590",
    "4413C000 591",
    "44140000 592",
    "44144000 593",
    "44148000 594",
    "4414C000 595",
    "44150000 596",
    "44154000 597",
    "44158000 598",
    "4415C000 599",
    "44160000 600",
    "44160000 6e2",
    "44164000 601",
    "44168000 602",
    "4416C000 603",
    "44170000 604",
    "44174000 605",
    "44178000 606",
    "4417C000 607",
    "44180000 608",
    "44184000 609",
    "44188000 610",
    "4418C000 611",
    "44190000 612",
    "44194000 613",
    "44198000 614",
    "4419C000 615",
    "441A0000 616",
    "441A4000 617",
    "441A8000 618",
    "441AC000 619",
    "441B0000 620",
    "441B4000 621",
    "441B8000 622",
    "441BC000 623",
    "441C0000 624",
    "441C4000 625",
    "441C8000 626",
    "441CC000 627",
    "441D0000 628",
    "441D4000 629",
    "441D8000 630",
    "441DC000 631",
    "441E0000 632",
    "441E4000 633",
    "441E8000 634",
    "441EC000 635",
    "441F0000 636",
    "441F4000 637",
    "441F8000 638",
    "441FC000 639",
    "44200000 640",
    "44204000 641",
    "44208000 642",
    "4420C000 643",
    "44210000 644",
    "44214000 645",
    "44218000 646",
    "4421C000 647",
    "44220000 648",
    "44224000 649",
    "44228000 650",
    "4422C000 651",
    "44230000 652",
    "44238000 654",
    "4423C000 655",
    "44240000 656",
    "44244000 657",
    "44248000 658",
    "44250000 660",
    "44254000 661",
    "44258000 662",
    "4425C000 663",
    "44260000 664",
    "44264000 665",
    "44268000 666",
    "4426C000 667",
    "44270000 668",
    "44278000 670",
    "4427C000 671",
    "44280000 672",
    "44284000 673",
    "44288000 674",
    "4428C000 675",
    "44294000 677",
    "44298000 678",
    "442A0000 680",
    "442A4000 681",
    "442B0000 684",
    "442B4000 685",
    "442BC000 687",
    "442C0000 688",
    "442C4000 689",
    "442C8000 690",
    "442CC000 691",
    "442D0000 692",
    "442D8000 694",
    "442DC000 695",
    "442E0000 696",
    "442E4000 697",
    "442E8000 698",
    "442F0000 700",
    "442F0000 7e2",
    "442F4000 701",
    "442F8000 702",
    "44300000 704",
    "44304000 705",
    "44308000 706",
    "4430C000 707",
    "44310000 708",
    "44314000 709",
    "44318000 710",
    "44318000 71e1",
    "4431C000 711",
    "44320000 712",
    "44324000 713",
    "44328000 714",
    "4432C000 715",
    "44330000 716",
    "44334000 717",
    "44338000 718",
    "4433C000 719",
    "44340000 720",
    "44344000 721",
    "44348000 722",
    "4434C000 723",
    "44350000 724",
    "44354000 725",
    "44358000 726",
    "4435C000 727",
    "44360000 728",
    "44368000 730",
    "4436C000 731",
    "44370000 732",
    "44374000 733",
    "44378000 734",
    "4437C000 735",
    "44380000 736",
    "44384000 737",
    "44388000 738",
    "44390000 740",
    "44394000 741",
    "44398000 742",
    "443A0000 744",
    "443A4000 745",
    "443A8000 746",
    "443AC000 747",
    "443B0000 748",
    "443B4000 749",
    "443B8000 750",
    "443B8000 75e01",
    "443BC000 751",
    "443C0000 752",
    "443C8000 754",
    "443CC000 755",
    "443D0000 756",
    "443D4000 757",
    "443D8000 758",
    "443E0000 760",
    "443E4000 761",
    "443EC000 763",
    "443F0000 764",
    "443F4000 765",
    "443F8000 766",
    "443FC000 767",
    "44400000 768",
    "44408000 770",
    "4440C000 771",
    "44410000 772",
    "44414000 773",
    "44418000 774",
    "4441C000 775",
    "44424000 777",
    "44428000 778",
    "44430000 780",
    "44434000 781",
    "44438000 782",
    "4443C000 783",
    "44440000 784",
    "44444000 785",
    "44448000 786",
    "4444C000 787",
    "44450000 788",
    "44458000 790",
    "4445C000 791",
    "4445C000 791e0",
    "44464000 793",
    "44468000 794",
    "4446C000 795",
    "44470000 796",
    "44474000 797",
    "44478000 798",
    "44480000 800",
    "44480000 8e2",
    "44484000 801",
    "44488000 802",
    "4448C000 803",
    "44490000 804",
    "44498000 806",
    "4449C000 807",
    "444A0000 808",
    "444A8000 810",
    "444AC000 811",
    "444B0000 812",
    "444B4000 813",
    "444B8000 814",
    "444C4000 817",
    "444C8000 818",
    "444D0000 820",
    "444D4000 821",
    "444DC000 823",
    "444E0000 824",
    "444E4000 825",
    "444E8000 826",
    "444EC000 827",
    "444F0000 828",
    "444F8000 830",
    "444FC000 831",
    "44500000 832",
    "44504000 833",
    "44508000 834",
    "4450C000 835",
    "44510000 836",
    "44514000 837",
    "44518000 838",
    "44520000 840",
    "44524000 841",
    "44528000 842",
    "4452C000 843",
    "44530000 844",
    "44538000 846",
    "4453C000 847",
    "44540000 848",
    "44544000 849",
    "44548000 850",
    "4454C000 851",
    "44554000 853",
    "44558000 854",
    "4455C000 855",
    "44564000 857",
    "44568000 858",
    "4456C000 859",
    "44570000 860",
    "44574000 861",
    "44578000 862",
    "4457C000 863",
    "44580000 864",
    "44584000 865",
    "44588000 866",
    "4458C000 867",
    "44590000 868",
    "44594000 869",
    "44598000 870",
    "4459C000 871",
    "445A0000 872",
    "445A4000 873",
    "445A8000 874",
    "445B0000 876",
    "445B4000 877",
    "445B8000 878",
    "445BC000 879",
    "445C0000 880",
    "445C4000 881",
    "445C8000 882",
    "445CC000 883",
    "445D0000 884",
    "445D8000 886",
    "445DC000 887",
    "445E0000 888",
    "445E4000 889",
    "445E8000 890",
    "445EC000 891",
    "445F0000 892",
    "445F4000 893",
    "445F8000 894",
    "44600000 896",
    "44604000 897",
    "44608000 898",
    "4460C000 899",
    "44610000 900",
    "44614000 901",
    "4461C000 903",
    "44620000 904",
    "44624000 905",
    "44628000 906",
    "4462C000 907",
    "44630000 908",
    "44638000 910",
    "4463C000 911",
    "44640000 912",
    "44644000 913",
    "44648000 914",
    "44650000 916",
    "44654000 917",
    "4465C000 919",
    "44660000 920",
    "44664000 921",
    "44668000 922",
    "4466C000 923",
    "44670000 924",
    "44678000 926",
    "4467C000 927",
    "44684000 929",
    "44688000 930",
    "44688000 93e1",
    "4468C000 931",
    "44694000 933",
    "44698000 934",
    "4469C000 935",
    "4469C000 935e00",
    "446A0000 936",
    "446A17B2 936.3702087402344",
    "446A4000 937",
    "446B0000 940",
    "446B4000 941",
    "446B8000 942",
    "446BC000 943",
    "446C0000 944",
    "446C4000 945",
    "446C8000 946",
    "446CC000 947",
    "446D0000 948",
    "446D4000 949",
    "446D8000 950",
    "446DC000 951",
    "446E0000 952",
    "446E4000 953",
    "446E8000 954",
    "446EC000 955",
    "446F0000 956",
    "446F4000 957",
    "44700000 960",
    "44704000 961",
    "4470C000 963",
    "44710000 964",
    "44718000 966",
    "4471C000 967",
    "44724000 969",
    "44728000 970",
    "4472C000 971",
    "44730000 972",
    "44734000 973",
    "44738000 974",
    "4473C000 975",
    "4473C000 975e0",
    "44740000 976",
    "44744000 977",
    "4474C000 979",
    "44750000 980",
    "44754000 981",
    "4475C000 983",
    "44760000 984",
    "44768000 986",
    "4476C000 987",
    "44774000 989",
    "44778000 990",
    "44784000 993",
    "44788000 994",
    "4478C000 995",
    "44790000 996",
    "44794000 997",
    "44798000 998",
    "4479C000 999",
    "447A0000 1000",
    "447A0000 1e3",
    "447AC000 1003",
    "447B0000 1004",
    "447B8000 1006",
    "447BC000 1007",
    "447C4000 1009",
    "447C8000 1010",
    "447D4000 1013",
    "447D8000 1014",
    "447E0000 1016",
    "447E4000 1017",
    "447EC000 1019",
    "447F0000 1020",
    "447F8000 1022",
    "447FC000 1023",
    "44800000 1024",
    "44804000 1026",
    "44806000 1027",
    "4480A000 1029",
    "4480C000 1030",
    "44812000 1033",
    "44814000 1034",
    "44818000 1036",
    "4481A000 1037",
    "4481E000 1039",
    "44820000 1040",
    "44826000 1043",
    "44828000 1044",
    "4482C000 1046",
    "4482E000 1047",
    "44832000 1049",
    "44834000 1050",
    "4483A000 1053",
    "4483C000 1054",
    "44840000 1056",
    "44842000 1057",
    "44846000 1059",
    "44848000 1060",
    "4484BB85 1061.86",
    "4484E000 1063",
    "44850000 1064",
    "44854000 1066",
    "44856000 1067",
    "4485A000 1069",
    "4485C000 1070",
    "44860000 1072",
    "44862000 1073",
    "44864000 1074",
    "44866000 1075",
    "44868000 1076",
    "4486A000 1077",
    "4486E000 1079",
    "44874000 1082",
    "4487C000 1086",
    "44882000 1089",
    "44886000 1091",
    "44888000 1092",
    "44890000 1096",
    "44896000 1099",
    "4489C000 1102",
    "448A4000 1106",
    "448AA000 1109",
    "448AC000 1110",
    "448AE000 1111",
    "448B0000 1112",
    "448B8000 1116",
    "448BE000 1119",
    "448C4000 1122",
    "448CC000 1126",
    "448D2000 1129",
    "448D8000 1132",
    "448E0000 1136",
    "44914000 1162",
    "44916000 1163",
    "44926000 1171",
    "4496E000 1207",
    "44986000 1219",
    "4499C000 1230",
    "449FE000 1279",
    "44A02000 1281",
    "44AB5DA6 1370.9265747070312",
    "44ADA000 1389",
    "44B12000 1417",
    "44B84000 1474",
    "44B8C000 1478",
    "44BCE000 1511",
    "44BFE000 1535",
    "44C2C000 1558",
    "44C36000 1563",
    "44C3C000 1566",
    "44C64000 1586",
    "44C8E000 1607",
    "44CD2000 1641",
    "44CFA000 1661",
    "44CFC000 1662",
    "44D08000 1668",
    "44D16000 1675",
    "44D6A000 1717",
    "44D90000 1736",
    "44DCA000 1765",
    "44DE6000 1779",
    "44E02000 1793",
    "44E28000 1812",
    "44E32000 1817",
    "44E5E000 1839",
    "44E6A000 1845",
    "44E84000 1858",
    "44E9E000 1871",
    "44EAA000 1877",
    "44EF8000 1916",
    "44F12000 1929",
    "44F26000 1939",
    "44F28000 1940",
    "44F3E000 1951",
    "44F42000 1953",
    "44F4C000 1958",
    "44F8E000 1991",
    "44F98000 1996",
    "44F9A000 1997",
    "44F9C000 1998",
    "44F9E000 1999",
    "44FA0000 2000",
    "44FA2000 2001",
    "44FA8000 2004",
    "44FC4148 2018.04",
    "44FC6000 2019",
    "44FC8000 2020",
    "44FC8148 2020.04",
    "44FD0000 2024",
    "44FE2000 2033",
    "44FEA000 2037",
    "44FFE000 2047",
    "45015000 2069",
    "4501E000 2078",
    "45027000 2087",
    "45032000 2098",
    "45069000 2153",
    "45072000 2162",
    "45077000 2167",
    "4508D000 2189",
    "450AE000 2222",
    "450C5000 2245",
    "450D6000 2262",
    "450E5000 2277",
    "450E8000 2280",
    "45100000 2304",
    "45113000 2323",
    "4512E000 2350",
    "4513C000 2364",
    "45158000 2392",
    "4515E000 2398",
    "45171000 2417",
    "45179000 2425",
    "4518E000 2446",
    "451A9000 2473",
    "451AA000 2474",
    "451C4000 25e2",
    "451C7000 2503",
    "451DD48C 2525.2840576171875",
    "451E1000 2529",
    "451FB000 2555",
    "45201000 2561",
    "45206000 2566",
    "4520C000 2572",
    "4523D000 2621",
    "45242000 2626",
    "4525F000 2655",
    "4528C000 27e2",
    "452A6000 2726",
    "452AE000 2734",
    "452B5000 2741",
    "452DD000 2781",
    "452EA000 2794",
    "452F0000 2800",
    "452F2000 2802",
    "452FA000 281e1",
    "45304000 2820",
    "45310000 2832",
    "4531D000 2845",
    "4531E000 2846",
    "45322000 2850",
    "4532E000 2862",
    "45331000 2865",
    "4533B000 2875",
    "45340000 288e1",
    "4534F000 2895",
    "4535A000 2906",
    "45378000 2936",
    "4537B000 2939",
    "45392000 2962",
    "45397000 2967",
    "4539E000 2974",
    "453A1000 2977",
    "453C6000 3014",
    "453D8000 3032",
    "453E6000 3046",
    "453EF000 3055",
    "453FA000 3066",
    "453FB000 3067",
    "45400000 3072",
    "45401000 3073",
    "45408000 3080",
    "4541E000 3102",
    "45427000 3111",
    "45428000 3112",
    "45435000 3125",
    "45447000 3143",
    "45487000 3207",
    "454A4000 3236",
    "454DD000 3293",
    "45505000 3333",
    "45512000 3346",
    "45520000 3360",
    "45548000 3400",
    "4555F000 3423",
    "45562000 3426",
    "45582000 3458",
    "45588000 3464",
    "4559B000 3483",
    "455BB000 3515",
    "455CC000 3532",
    "455CF000 3535",
    "45613000 3603",
    "45618000 3608",
    "45675000 3701",
    "4568F000 3727",
    "456D0000 3792",
    "45701000 3841",
    "45723000 3875",
    "45744000 3908",
    "45759000 3929",
    "4578A000 3978",
    "45795000 3989",
    "4579F000 3999",
    "457A0000 4000",
    "457A0000 4e3",
    "457AE000 4014",
    "457BD000 4029",
    "457E1000 4065",
    "457E2000 4066",
    "457F0000 4080",
    "457FF000 4095",
    "45800000 4096",
    "4580E000 4124",
    "45814000 4136",
    "4582B800 4183",
    "4582C000 4184",
    "4582E800 4189",
    "4582F800 4191",
    "45834000 4200",
    "45838800 4209",
    "45841000 4226",
    "4584B800 4247",
    "45864000 4296",
    "45875000 4330",
    "4587E800 4349",
    "45890000 4384",
    "45898000 4400",
    "458A5800 4427",
    "458AC000 4440",
    "458AE000 4444",
    "458B3800 4455",
    "458D7000 4526",
    "458ED800 4571",
    "458FA800 4597",
    "458FB000 4598",
    "45902800 4613",
    "45904800 4617",
    "4590C000 4632",
    "45915800 4651",
    "4591A800 4661",
    "4591C800 4665",
    "45921000 4674",
    "45925000 4682",
    "4592E000 4700",
    "45934800 4713",
    "45939800 4723",
    "4593A000 4724",
    "45948800 4753",
    "45954800 4777",
    "45965800 4811",
    "4596A000 4820",
    "45973000 4838",
    "45993000 4902",
    "45999800 4915",
    "4599E800 4925",
    "4599F000 4926",
    "459A0800 4929",
    "459B6000 4972",
    "459B8000 4976",
    "459BF800 4991",
    "459CB000 5014",
    "459CE800 5021",
    "459E9000 5074",
    "459F0000 5088",
    "459FC000 5112",
    "459FD000 5114",
    "45A10800 5153",
    "45A18800 5169",
    "45A1D000 5178",
    "45A1E000 5180",
    "45A29000 5202",
    "45A38000 5232",
    "45A3A000 5236",
    "45A58000 5296",
    "45A5A800 5301",
    "45A5E000 5308",
    "45A66000 5324",
    "45A6B000 5334",
    "45A77800 5359",
    "45A83000 5382",
    "45A8C000 54e2",
    "45A93800 5415",
    "45A97800 5423",
    "45AA0800 5441",
    "45AC4800 5513",
    "45ACF800 5535",
    "45AD9800 5555",
    "45ADC000 5560",
    "45AE7800 5583",
    "45AE9000 5586",
    "45AF0000 5600",
    "45AF0800 5601",
    "45AF2800 5605",
    "45B02000 5636",
    "45B06800 5645",
    "45B1D800 5691",
    "45B21000 5698",
    "45B42000 5764",
    "45B5B000 5814",
    "45B5F800 5823",
    "45B6F000 5854",
    "45B7C000 5880",
    "45B85800 5899",
    "45B8E000 5916",
    "45BA4000 5960",
    "45BAD800 5979",
    "45BB7000 5998",
    "45BBC800 6009",
    "45BC1800 6019",
    "45BC9000 6034",
    "45BCD000 6042",
    "45BD2800 6053",
    "45BE2000 6084",
    "45BF5000 6122",
    "45C03000 6150",
    "45C11000 6178",
    "45C17800 6191",
    "45C20800 6209",
    "45C25000 6218",
    "45C2F800 6239",
    "45C3B000 6262",
    "45C41000 6274",
    "45C51000 6306",
    "45C53800 6311",
    "45C574A4 6318.580322265625",
    "45C5C800 6329",
    "45C5E000 6332",
    "45C6C000 6360",
    "45C7E800 6397",
    "45C7F000 6398",
    "45C7F800 6399",
    "45C82800 6405",
    "45C84800 6409",
    "45C89800 6419",
    "45C8A000 642e1",
    "45C9B800 6455",
    "45C9F000 6462",
    "45CA8800 6481",
    "45CAF000 6494",
    "45CB2000 6500",
    "45CBC800 6521",
    "45CC1000 653e1",
    "45CC5000 6538",
    "45CD2800 6565",
    "45CEC800 6617",
    "45D01000 6658",
    "45D05000 6666",
    "45D10000 6688",
    "45D20000 6720",
    "45D2B000 6742",
    "45D40000 6784",
    "45D45000 6794",
    "45D48000 6800",
    "45D58000 6832",
    "45D6C000 6872",
    "45D85800 6923",
    "45D97000 6958",
    "45DAA800 6997",
    "45DAC000 7e3",
    "45DAF800 7007",
    "45DB8800 7025",
    "45DC6000 7052",
    "45DC6800 7053",
    "45DD8000 7088",
    "45DE2800 7109",
    "45DE4000 7112",
    "45DF2800 7141",
    "45E04800 7177",
    "45E09800 7187",
    "45E18000 7216",
    "45E39000 7282",
    "45E3B000 7286",
    "45E57800 7343",
    "45E5C000 7352",
    "45E68800 7377",
    "45E70000 7392",
    "45E74000 7400",
    "45E74800 7401",
    "45E7B800 7415",
    "45E7C000 7416",
    "45E7F000 7422",
    "45E80800 7425",
    "45E89800 7443",
    "45E8A000 7444",
    "45E9B000 7478",
    "45EA6800 7501",
    "45EA7800 7503",
    "45EB7800 7535",
    "45EBE000 7548",
    "45EC8800 7569",
    "45ECB800 7575",
    "45EDA000 7604",
    "45EE2800 7621",
    "45EE5000 7626",
    "45EEC800 7641",
    "45EEF800 7647",
    "45EF0000 7648",
    "45F0E800 7709",
    "45F0F800 7711",
    "45F20000 7744",
    "45F22800 7749",
    "45F2C000 7768",
    "45F30800 7777",
    "45F35800 7787",
    "45F39800 7795",
    "45F41000 7810",
    "45F45000 7818",
    "45F5C000 7864",
    "45F64000 7880",
    "45F8B000 7958",
    "45F8B800 7959",
    "45F96000 7980",
    "45FA0000 8000",
    "45FA0800 8001",
    "45FB3000 8038",
    "45FBC800 8057",
    "45FC6800 8077",
    "45FCB800 8087",
    "45FD4800 8105",
    "45FD7800 8111",
    "45FE0000 8128",
    "45FEF000 8158",
    "45FF0800 8161",
    "46004800 8210",
    "46005400 8213",
    "4600F000 8252",
    "4601B000 8300",
    "46021800 8326",
    "46025400 8341",
    "46026800 8346",
    "46027400 8349",
    "4602C800 8370",
    "4602E400 8377",
    "4602F000 8380",
    "46032400 8393",
    "46032800 8394",
    "46033000 8396",
    "46035800 8406",
    "46037000 8412",
    "4603B800 8430",
    "46043400 8461",
    "4604B400 8493",
    "4604B800 8494",
    "46050400 8513",
    "4605A000 8552",
    "46060400 8577",
    "46065C00 8599",
    "46066C00 8603",
    "46069000 8612",
    "46069400 8613",
    "4606A400 8617",
    "4606F400 8637",
    "46071000 8644",
    "4607D000 8692",
    "4607E000 8696",
    "46082800 8714",
    "4608E800 8762",
    "46092C00 8779",
    "4609F400 8829",
    "460AE000 8888",
    "460AFC00 8895",
    "460B6000 8920",
    "460B6400 8921",
    "460C1400 8965",
    "460C2C00 8971",
    "460CA000 9000",
    "460CA000 9e3",
    "460CEC00 9019",
    "460CFC00 9023",
    "460D4C00 9043",
    "460E2000 9096",
    "460F9000 9188",
    "46107800 9246",
    "4610C400 9265",
    "4610DC00 9271",
    "4610E400 9273",
    "46113000 9292",
    "46119000 9316",
    "4612C000 9392",
    "4612D400 9397",
    "46135800 9430",
    "46137000 9436",
    "46140000 9472",
    "46141400 9477",
    "46142000 9480",
    "46143400 9485",
    "46143800 9486",
    "46144C00 9491",
    "46145C00 9495",
    "46147800 9502",
    "46148800 9506",
    "46155800 9558",
    "46165C00 9623",
    "46166000 9624",
    "46171800 9670",
    "46172000 9672",
    "46181000 9732",
    "46184400 9745",
    "4618F400 9789",
    "461A9000 9892",
    "461AE800 9914",
    "461B5800 9942",
    "461BC000 9968",
    "461BEC00 9979",
    "461C1C00 9991",
    "461C3C00 9999",
    "461C4000 10000",
    "461C4000 1e4",
    "461E6400 10137",
    "4620B000 10284",
    "46258800 10594",
    "462D9C00 11111",
    "46333C00 11471",
    "4634FC00 11583",
    "463F2000 12232",
    "46426800 12442",
    "4647EC00 12795",
    "464BF000 13052",
    "46506000 13336",
    "46592000 13896",
    "465CB400 14125",
    "46634400 14545",
    "4663F000 14588",
    "466E9C00 15271",
    "46722974 15498.36376953125",
    "46742400 15625",
    "4677D800 15862",
    "467D3800 16206",
    "467FF800 16382",
    "46802FE4 16407.9462890625",
    "46816200 16561",
    "4681C200 16609",
    "4684D000 17e3",
    "4688174C 17419.6494140625",
    "468AA400 17746",
    "46912800 18580",
    "4693A600 18899",
    "4694F800 19068",
    "4695B200 19161",
    "46984600 19491",
    "469C3E00 19999",
    "469CAC00 20054",
    "469CE400 20082",
    "469E2E00 20247",
    "469E6E00 20279",
    "46A23A00 20765",
    "46A5AE00 21207",
    "46A9BE00 21727",
    "46AB9800 21964",
    "46AD9C00 22222",
    "46BA7600 23867",
    "46BD2200 24209",
    "46BF0800 24452",
    "46C40000 25088",
    "46C80000 256e2",
    "46C9BE00 25823",
    "46CA1400 25866",
    "46CAD200 25961",
    "46CC8E00 26183",
    "46CEA600 26451",
    "46D15A00 26797",
    "46D1A800 26836",
    "46D4EE00 27255",
    "46D54600 27299",
    "46DAAA00 27989",
    "46DB4A00 28069",
    "46DD1800 28300",
    "46E53600 29339",
    "46EBA600 30163",
    "46EBDC00 30190",
    "46EC162A 30219.0830078125",
    "46F28A00 31045",
    "46F68E00 31559",
    "46F6C600 31587",
    "46F78000 31680",
    "47010C00 33036",
    "4701C500 33221",
    "47023500 33333",
    "4703B900 33721",
    "4703D100 33745",
    "47074300 34627",
    "4707FA00 34810",
    "47091700 35095",
    "470A6B00 35435",
    "470D1100 36113",
    "470D8000 36224",
    "470E0E00 36366",
    "470E7F00 36479",
    "470FE500 36837",
    "47109600 37014",
    "4710E000 37088",
    "4713A900 37801",
    "4713A900 37801e0",
    "47160D00 38413",
    "471ABD00 39613",
    "471AF900 39673",
    "471C4000 40000",
    "471C4000 4e4",
    "471C8200 40066",
    "471E4300 40515",
    "471F8000 40832",
    "47218200 41346",
    "4722D600 41686",
    "47233300 41779",
    "47241000 42e3",
    "47269500 42645",
    "4726D700 42711",
    "47278900 42889",
    "47278E00 42894",
    "47291700 43287",
    "47291E00 43294",
    "472B330C 43827.048828125",
    "472CEF00 44271",
    "472D9C00 44444",
    "472ED100 44753",
    "4732F500 45813",
    "4735B600 46518",
    "47370B00 46859",
    "47382A00 47146",
    "47395500 47445",
    "47397F00 47487",
    "4739CC00 47564",
    "4739DB00 47579",
    "473F9500 49045",
    "47426500 49765",
    "47435000 5e4",
    "47448100 50305",
    "4746B000 50864",
    "47484300 51267",
    "4748B900 51385",
    "47491500 51477",
    "47495300 51539",
    "474A6F00 51823",
    "474A8C00 51852",
    "474B6000 52064",
    "474BE000 52192",
    "474BF200 52210",
    "474C8B00 52363",
    "474D0100 52481",
    "47538600 54150",
    "47547A00 54394",
    "47559600 54678",
    "4755F000 54768",
    "47571B00 55067",
    "47583300 55347",
    "47584800 55368",
    "4758A800 55464",
    "47590300 55555",
    "47595500 55637",
    "47596500 55653",
    "475AC800 56008",
    "475B2900 56105",
    "475B9C00 56220",
    "475D8800 56712",
    "475DD600 56790",
    "475F9C00 57244",
    "475FF000 57328",
    "47606C00 57452",
    "47615F00 57695",
    "4762EB00 58091",
    "47634200 58178",
    "47634400 58180",
    "47635800 582e2",
    "47652400 58660",
    "476A2800 59944",
    "476A6000 6e4",
    "476C4F00 60495",
    "476D1400 60692",
    "476E4800 61E3",
    "476F6100 61281",
    "476F8100 61313",
    "47735400 62292",
    "47749800 62616",
    "47757400 62836",
    "4775AE00 62894",
    "4777C100 63425",
    "4777D000 63440",
    "47780D00 63501",
    "47796D00 63853",
    "477AB100 64177",
    "477AE600 6423e1",
    "477B1B00 64283",
    "477B4F00 64335",
    "477CC800 64712",
    "477D4800 64840",
    "477F6C00 65388",
    "47800000 65536",
    "47809280 65829",
    "47810E00 66076",
    "47820A00 66580",
    "47822500 66634",
    "47823500 66666",
    "4782DF00 67006",
    "4785F400 68584",
    "47860E80 68637",
    "4787DC80 69561",
    "4787ED00 69594",
    "4788BE80 70013",
    "47894C80 70297",
    "4789D280 70565",
    "478A6500 70858",
    "478A7000 7088e1",
    "478B8400 71432",
    "478C2700 71758",
    "478D1F80 72255",
    "478D3800 72304",
    "478D9880 72497",
    "478DA200 72516",
    "478FC480 73609",
    "478FF400 73704",
    "47903400 73832",
    "4790F980 74227",
    "47928F00 75038",
    "47938980 75539",
    "4793F780 75759",
    "47941080 75809",
    "47949300 76070",
    "4794E900 76242",
    "47959680 76589",
    "4796B600 77164",
    "47975680 77485",
    "4797CB80 77719",
    "4797E880 77777",
    "47980600 77836",
    "47989680 78125",
    "4798B500 78186",
    "479A0080 78849",
    "479A6F80 79071",
    "479AF900 79346",
    "479BA380 79687",
    "479BB400 79720",
    "479C4000 80000",
    "479C4000 8e4",
    "479C5580 80043",
    "479D0400 80392",
    "479D1E00 80444",
    "479D8E00 80668",
    "479F1F80 81471",
    "479FD900 81842",
    "47A1C100 82818",
    "47A20A00 82964",
    "47A25B00 83126",
    "47A29A80 83253",
    "47A2E200 83396",
    "47A33000 83552",
    "47A33080 83553",
    "47A33E80 83581",
    "47A34080 83585",
    "47A53000 84576",
    "47A56780 84687",
    "47A5FD80 84987",
    "47A605A0 85003.24609375",
    "47A64680 85133",
    "47A7C900 85906",
    "47A7CE00 85916",
    "47A85680 86189",
    "47A8B400 86376",
    "47A9C680 86925",
    "47AA4E00 87196",
    "47AA7580 87275",
    "47AAB000 87392",
    "47AB2480 87625",
    "47AD5580 88747",
    "47AD7680 88813",
    "47AD9C00 88888",
    "47ADBF00 88958",
    "47ADE600 89036",
    "47AEED00 89562",
    "47AF3480 89705",
    "47AF7680 89837",
    "47AFC800 9e4",
    "47AFF000 90080",
    "47B0E180 90563",
    "47B12D80 90715",
    "47B15D80 90811",
    "47B18000 90880",
    "47B18900 90898",
    "47B27800 91376",
    "47B28D80 91419",
    "47B2B780 91503",
    "47B3E780 92111",
    "47B4A380 92487",
    "47B52500 92746",
    "47B68280 93445",
    "47B78200 93956",
    "47B8ED00 94682",
    "47B96900 94930",
    "47BA2200 953e2",
    "47BA2A80 95317",
    "47BA4200 95364",
    "47BA9380 95527",
    "47BB5C00 95928",
    "47BC0580 96267",
    "47BC0800 96272",
    "47BDAB80 97111",
    "47BDE500 97226",
    "47BE0200 97284",
    "47BEDD80 97723",
    "47BEDE00 97724",
    "47BEEF00 97758",
    "47BF6800 98e3",
    "47BF8A00 98068",
    "47BFBA80 98165",
    "47BFBE00 98172",
    "47C0DD00 98746",
    "47C16580 99019",
    "47C19180 99107",
    "47C26A80 99541",
    "47C2EA80 99797",
    "47C34D80 99995",
    "47C34F80 99999",
    "47C35000 100000",
    "47C35000 1e5",
    "47C65E00 101564",
    "47C8DE00 102844",
    "47D2E280 107973",
    "47D90380 111111",
    "47DEA800 114000",
    "47FA0300 128006",
    "47FFA500 130890",
    "4807C540 139029",
    "480F2B80 146606",
    "48102400 1476e2",
    "4810F600 148440",
    "48149A80 152170",
    "48191B84 156782.0703125",
    "4819D280 157514",
    "481AE000 158592",
    "481B0100 158724",
    "4825DC00 169840",
    "482C7B00 176620",
    "482C9C00 176752",
    "482DC6C0 177947",
    "482E3800 178400",
    "483B2080 191618",
    "483EC680 195354",
    "483F90C0 196163",
    "48435000 200000",
    "48435000 20e04",
    "48435000 2e5",
    "48451A40 201833",
    "48464BC0 203055",
    "484CDEC0 209787",
    "4852D6C0 215899",
    "48567FC0 219647",
    "48568100 219652",
    "48590380 222222",
    "485919C0 222311",
    "485CED40 226229",
    "485D93C0 226895",
    "485E62C0 227723",
    "486003C0 229391",
    "486805C0 237591",
    "486CD0C0 242499",
    "486ED800 244576",
    "48701040 245825",
    "4870F0C0 246723",
    "487319C0 248935",
    "48796E00 255416",
    "487A5C40 256369",
    "487C9A40 258665",
    "487CCD40 258869",
    "488131A0 264589",
    "48850A60 272467",
    "48885FC0 279294",
    "4888B800 280000",
    "488D6300 289560",
    "488DB840 290242",
    "488E8260 291859",
    "4891A980 298316",
    "48927B60 299995",
    "48933120 301449",
    "4895D6A0 306869",
    "4896C580 308780",
    "48997AA0 314325",
    "489A6DE0 316271",
    "489E61A0 324365",
    "489F7480 326564",
    "48A057B0 328381.484375",
    "48A0AE80 329076",
    "48A269A0 332621",
    "48A2C2A0 333333",
    "48A60400 340000",
    "48A8C580 345644",
    "48BEBC20 390625",
    "48C036C0 393654",
    "48C1D5A0 396973",
    "48C2E060 399107",
    "48C35000 400000",
    "48C35000 4e5",
    "48C3A380 400668",
    "48C40980 401484",
    "48C71160 407691",
    "48C83200 410000",
    "48D00460 426019",
    "48D228C0 430406",
    "48D38360 433179",
    "48D48960 435275",
    "48D635E0 438703",
    "48D90380 444444",
    "48D948E0 444999",
    "48D96B60 445275",
    "48DBD640 450226",
    "48DDAD40 453994",
    "48E0D200 460432",
    "48E4A8A0 468293",
    "48E82E40 475506",
    "48E851E0 475791",
    "48E983C0 478238",
    "48E99AE0 478423",
    "48EAEE40 481138",
    "48EBC260 482835",
    "48EC4000 483840",
    "48ED3860 485827",
    "48F022C0 491798",
    "48F42400 50e4",
    "48F42400 5e5",
    "48F98480 511012",
    "48FA53C0 512670",
    "49037740 538484",
    "490433D0 541501",
    "49056290 546345",
    "49063B70 549815",
    "4907A230 555555",
    "490BF660 573286",
    "490C0000 573440",
    "490CDE90 577001",
    "490E2C20 582338",
    "490E9E60 584166",
    "490EDB30 585139",
    "490F8490 587849",
    "490FDAE0 589230",
    "49103830 590723",
    "49115190 595225",
    "49117750 595829",
    "49131000 602368",
    "49146000 607744",
    "49159960 612758",
    "491653C0 615740",
    "4916A280 617e3",
    "49175E00 620000",
    "49181480 622920",
    "491A9CD0 633293",
    "491AB520 633682",
    "491AEC40 634564",
    "491B21E0 635422",
    "491F74B0 653131",
    "49209B20 657842",
    "4920D8C0 658828",
    "4920DCC0 658892",
    "49215420 660802",
    "492191F0 661791",
    "4922AE70 666343",
    "4922C2A0 666666",
    "492349D0 668829",
    "4925BD00 678864",
    "492606B0 680043",
    "4927A1B0 686619",
    "4927F750 687989",
    "49298D70 694487",
    "492A0290 696361",
    "492A1C60 696774",
    "492A2830 696963",
    "492A4300 697392",
    "492AE600 700000",
    "492AE600 7e5",
    "492C5C80 705992",
    "492DB5B0 711515",
    "49302EA0 721642",
    "4932C820 732290",
    "493380C0 735244",
    "4935A440 744004",
    "4936FAD0 749485",
    "49389CE0 756174",
    "493A2360 762422",
    "493AA4A0 764490",
    "493B55F0 767327",
    "493B8740 768116",
    "493B8E20 768226",
    "493C8F30 772339",
    "493D3580 775e3",
    "493DE310 777777",
    "493E6E00 78e4",
    "493EFB64 782262.28125",
    "493F7D10 784337",
    "493FEE70 786151",
    "494185B0 792667",
    "4941BD30 793555",
    "49423260 795430",
    "49424B60 795830",
    "49435000 800000",
    "49435000 8e5",
    "4944DD50 806357",
    "49452670 807527",
    "4947F330 818995",
    "49483180 819992",
    "494A5030 828675",
    "494AB4B0 830283",
    "494AC540 830548",
    "494CAD10 838353",
    "494DF5A0 843610",
    "494EDCB0 847307",
    "494F0490 847945",
    "495110E0 856334",
    "4951AA60 858790",
    "49531EE0 864750",
    "49535E40 865764",
    "4953BA90 867241",
    "49552280 873e03",
    "49555360 873782",
    "4956A490 879177",
    "4957DA60 884134",
    "4958C540 887892",
    "4958D480 888136",
    "49590380 888888",
    "49595FA0 890362",
    "49596790 890489",
    "4959F3B0 892731",
    "495A9010 895233",
    "495B05C0 897116",
    "495B2FD0 897789",
    "495B9B00 899504",
    "495BBA00 9e5",
    "495DD720 908658",
    "495DE0A0 908810",
    "495E7330 911155",
    "495F2100 913936",
    "495FE390 917049",
    "4961CFB0 924923",
    "4962A260 928294",
    "49641970 934295",
    "496491B0 936219",
    "49649330 936243",
    "49662770 942711",
    "496719E0 946590",
    "49672880 946824",
    "496753E0 947518",
    "4967A220 948770",
    "496981F0 956447",
    "496A1A50 958885",
    "496B43E0 963646e0",
    "496C5E30 968163",
    "496D16F0 971119",
    "496F1920 979346",
    "496FB1B0 981787",
    "49708740 985204",
    "4970B3B0 985915",
    "4971AEF0 989935",
    "49733480 996168",
    "4973EB20 999090",
    "497423F0 999999",
    "49742400 1000000",
    "49742400 1e6",
    "497484E0 1001550",
    "49784FE0 1017086",
    "49800000 1048576",
    "49824680 1067216",
    "49834C80 1075600",
    "49849078 1085967",
    "4987A238 1111111",
    "498A6AB8 1133911",
    "498AA678 1135823",
    "49909E68 1184717",
    "49918F98 1192435",
    "49920760 1196268",
    "4996AD40 1234344",
    "49999C58 1258379",
    "499B6E60 1273292",
    "49AA9EF0 1397726",
    "49ACE2E0 1416284",
    "49AFEB48 1441129",
    "49B01F70 1442798",
    "49B87262 1510988.3125",
    "49B95070 1518094",
    "49BEC880 1562896",
    "49C04738 1575143",
    "49C06B60 15763e2",
    "49C9B478 1652367",
    "49D05330 1706598",
    "49D17398 1715827",
    "49E21D50 1852330",
    "49E5E080 1883152",
    "49E622C0 1885272",
    "49E97628 1912517",
    "49EB0F78 1925615",
    "49ECF008 1940993",
    "49EE6B28 1953125",
    "49F07B58 1970027",
    "49F42400 2000000",
    "49F42400 2e6",
    "49F83168 2033197",
    "49F9A7F0 2045182",
    "4A00BFF0 2109436",
    "4A026540 2136400",
    "4A033A6C 2150043",
    "4A04B4A0 2174248",
    "4A060400 2195712",
    "4A07A238 2222222",
    "4A07BF90 22241e2",
    "4A07DC94 2225957",
    "4A086E50 2235284",
    "4A0BBCA0 2289448",
    "4A0C6D5C 2300759",
    "4A0C8A60 2302616",
    "4A0CE124 2308169",
    "4A0DFAA8 2326186",
    "4A0EBF40 2338768",
    "4A0F78D0 2350644",
    "4A11B7F8 2387454",
    "4A122A1C 2394759",
    "4A133D40 2412368",
    "4A1801B4 2490477",
    "4A1847A8 2494954",
    "4A1CBA34 2567821",
    "4A1DF0A0 2587688",
    "4A1E41C8 2592882",
    "4A1EB100 26e5",
    "4A205244 2626705",
    "4A21E044 2652177",
    "4A26648C 2726179",
    "4A2AC054 2797589",
    "4A2CB0BC 2829359",
    "4A2CD31C 2831559",
    "4A31197C 2901599",
    "4A348270 2957468",
    "4A34BF94 2961381",
    "4A35E3BC 2980079",
    "4A360ACC 2982579",
    "4A371B00 3e6",
    "4A3A8664 3056025",
    "4A3A9FBC 3057647",
    "4A3DC308 3109058",
    "4A3FD4D8 3142966",
    "4A4542A0 3231912",
    "4A455580 3233120",
    "4A4588C0 3236400",
    "4A4789C8 3269234",
    "4A4A4C30 3314444",
    "4A4B7354 3333333",
    "4A4E0AB0 3375788",
    "4A4FB144 3402833",
    "4A559F80 3500000",
    "4A57F910 35385e2",
    "4A587710 3546564",
    "4A59A12C 3565643",
    "4A5B02BC 3588271",
    "4A5CA22C 3614859",
    "4A61D204 3699841",
    "4A643160 3738712",
    "4A64E50C 3750211",
    "4A69C664 3830169",
    "4A6AEC60 3848984",
    "4A6BC828 3863050",
    "4A6C4990 3871332",
    "4A6D1F44 3885009",
    "4A6D446C 3887387",
    "4A6E0D58 3900245.875",
    "4A6E7C80 3907360",
    "4A6EE6D0 3914164",
    "4A6FB810 3927556",
    "4A71C760 3961304",
    "4A72DAE4 3978937",
    "4A73BFD0 3993588",
    "4A742400 4000000",
    "4A742400 40e5",
    "4A764350 4034772",
    "4A76A054 4040725",
    "4A76F100 4045888",
    "4A77067C 4047263",
    "4A78EF20 4078536",
    "4A7977F0 4087292",
    "4A7A4070 4100124",
    "4A7D6CAC 4152107",
    "4A7EA8E4 4172345",
    "4A7F9270 4187292",
    "4A7FA2A8 4188330",
    "4A7FBBC0 4189936",
    "4A809182 4212929",
    "4A80C4C6 4219491",
    "4A816500 4240000",
    "4A81B3C2 4250081",
    "4A83981E 4312079",
    "4A83EE56 4323115",
    "4A8475E6 4340467",
    "4A852C38 4363804",
    "4A8535E2 4365041",
    "4A869D78 4411068",
    "4A86F0C4 4421730",
    "4A879868 4443188",
    "4A87A238 4444444",
    "4A87A5D4 4444906",
    "4A8A8CC0 454e4",
    "4A8AA648 4543268",
    "4A8C4492 4596297",
    "4A8C6180 46e5",
    "4A8C7308 4602244",
    "4A8CBC8A 4611653",
    "4A8CE696 4617035",
    "4A8EDADE 4681071",
    "4A909D22 4738705",
    "4A93186E 4820023",
    "4A95AD2C 4904598",
    "4A95C034 4907034",
    "4A96488A 4924485",
    "4A967D0A 4931205",
    "4A96D020 4941840",
    "4A97A78C 4969414",
    "4A9805CC 4981478",
    "4A989680 5000000",
    "4A989680 5e6",
    "4A98A126 5001363",
    "4A997A56 5029163",
    "4A9C0BE2 5113329",
    "4A9C4A28 51213e2",
    "4A9D2884 5149762",
    "4A9D3C3E 5152287",
    "4A9E6AC2 5191009",
    "4A9EC846 5202979",
    "4A9F9B60 523e4",
    "4AA26E2E 5322519.25",
    "4AA2AC4E 5330471",
    "4AA3F010 5371912",
    "4AA4342C 5380630",
    "4AA47D50 5389992",
    "4AA567C0 542e4",
    "4AA5BE9A 5431117",
    "4AA851B4 5515482",
    "4AA98AC6 5555555",
    "4AAA8446 5587491",
    "4AAADEDE 5599087",
    "4AAB717A 5617853",
    "4AABA8FC 5624958",
    "4AABAEB6 5625691",
    "4AAC1C5C 5639726",
    "4AAC5638 5647132",
    "4AAC60CC 5648486",
    "4AADA0CC 5689446",
    "4AAEBEC2 5726049",
    "4AAF5F22 5746577",
    "4AB1200E 5804039",
    "4AB2AC30 5854744",
    "4AB365BC 5878494",
    "4AB42BD6 5903851",
    "4AB4AA00 592e4",
    "4AB5D47A 5958205",
    "4AB5F5C0 5962464",
    "4AB624EA 5968501",
    "4AB67498 59787e2",
    "4AB67EC0 598e4",
    "4AB6EA94 5993802",
    "4AB72C96 6002251",
    "4AB8778E 6044615",
    "4AB8A82A 6050837",
    "4AB8F47C 6060606",
    "4ABAC962 6120625",
    "4ABAF6A4 6126418",
    "4ABE7654 6241066",
    "4ABEED48 6256292",
    "4AC0B474 6314554",
    "4AC3864A 6406949",
    "4AC4FFF8 6455292",
    "4AC53036 6461467",
    "4AC6ED2E 6518423",
    "4AC79534 6539930",
    "4AC7B6D8 6544236",
    "4AC80002 6553601",
    "4AC8C9A6 6579411",
    "4AC95AAA 6597973",
    "4ACAD5E2 6646513",
    "4ACB7354 6666666",
    "4ACD8D7A 6735549",
    "4ACDB482 6740545",
    "4ACE8B80 6768064",
    "4ACECEAE 6776663",
    "4ACF4680 6792e3",
    "4AD21726 6884243",
    "4AD35CA6 6925907",
    "4AD391DA 6932717",
    "4AD3FA36 6946075",
    "4AD449CA 6956261",
    "4AD4689E 6960207",
    "4AD5562C 6990614",
    "4AD59F80 7e6",
    "4AD5AA46 7001379",
    "4AD79484 7064130",
    "4AD7F70C 7076742",
    "4AD93EB4 7118682",
    "4ADA6FE0 7157744",
    "4ADAEC78 7173692",
    "4ADB2E88 7182148",
    "4ADC8C1E 7226895",
    "4ADE9354 7293354",
    "4ADEECEE 7304823",
    "4ADF6380 7320000",
    "4AE0F386 7371203",
    "4AE13F14 7380874",
    "4AE21AA8 7408980",
    "4AE270D6 7420011",
    "4AE35622 7449361",
    "4AE3E6E2 7467889",
    "4AE44FB4 7481306",
    "4AE52A58 7509292",
    "4AE58934 7521434",
    "4AE664B4 7549530",
    "4AE88AB0 7619928",
    "4AE93032 7641113",
    "4AEA3A4E 7675175",
    "4AED5BE2 7777777",
    "4AEE1C9E 7802447",
    "4AEF0AD0 7832936",
    "4AF1EA78 7927100",
    "4AF1FA8C 7929158",
    "4AF29330 7948696",
    "4AF315B8 7965404",
    "4AF3478A 7971781",
    "4AF42400 8000000",
    "4AF42400 80e5",
    "4AF42400 8e6",
    "4AF5FD28 8060564",
    "4AF6C0A4 8085586",
    "4AF6FB7E 8093119",
    "4AF8F0A8 8157268",
    "4AF8F8D6 8158315",
    "4AF93C48 8166948",
    "4AFC99DE 8277231",
    "4AFD5B0C 8301958",
    "4AFE6200 8335616",
    "4AFFBF4A 8380325",
    "4AFFEEB6 8386395",
    "4B000006 8388614.5",
    "4B00828C 8422028",
    "4B011844 8460356",
    "4B01407C 8470652",
    "4B02273A 8529722",
    "4B022BC6 8530886",
    "4B023525 8533285",
    "4B0250EA 8540394",
    "4B031558 8590680",
    "4B03AE76 8629878",
    "4B045588 8672648",
    "4B053590 873e4",
    "4B053BC3 8731587",
    "4B06CC74 8834164",
    "4B0746D8 8865496",
    "4B0767DC 8873948",
    "4B07A238 8888888",
    "4B07B00F 8892431",
    "4B086DB8 8940984",
    "4B089CC7 8953031",
    "4B08C382 8962946",
    "4B090452 8979538",
    "4B092DFD 8990205",
    "4B094C1B 8997915",
    "4B095440 90e5",
    "4B0A74EA 9073898",
    "4B0AD089 9097353",
    "4B0B24CD 9118925",
    "4B0B5716 9131798",
    "4B0B7E9B 9141915",
    "4B0CE24D 9232973",
    "4B0DA6B5 9283253",
    "4B0F1B27 9378599",
    "4B0F27F1 9381873",
    "4B0F6EC0 94e5",
    "4B1026B5 9447093",
    "4B1039CC 9451980",
    "4B10AECF 9481935",
    "4B1122D6 9511638",
    "4B116CC6 9530566",
    "4B11B8B0 955e4",
    "4B11F652 9565778",
    "4B123145 9580869",
    "4B138D70 967e4",
    "4B139EFD 9674493",
    "4B13F52A 9696554",
    "4B143BC8 9714632",
    "4B1454CF 9721039",
    "4B14A8ED 9742573",
    "4B1502F9 9765625",
    "4B154D15 9784597",
    "4B15A07C 9805948",
    "4B15BCBA 9813178",
    "4B15DE34 9821748",
    "4B163AAA 9845418",
    "4B16ABAB 9874347",
    "4B16DE4B 9887307",
    "4B16F230 9892400",
    "4B1702D3 9896659",
    "4B170F0D 9899789",
    "4B174328 9913128",
    "4B175BA0 9919392",
    "4B1767F3 9922547",
    "4B176B01 9923329",
    "4B17EB48 9956168",
    "4B1877F8 9992184",
    "4B18967F 9999999",
    "4B189680 1e7",
    "4B1A1A17 10099223",
    "4B24CA65 10799717",
    "4B298AC7 11111111",
    "4B30A3FF 11576319",
    "4B317911 11630865",
    "4B3840E2 12075234",
    "4B40A3B2 12624818",
    "4B40F04B 12644427",
    "4B41CE98 12701336",
    "4B437F39 12812089",
    "4B4698B8 13015224",
    "4B472556 13051222",
    "4B47D6E3 13096675",
    "4B4891A0 13144480",
    "4B519C31 13737009",
    "4B528654 13796948",
    "4B59132F 14226223",
    "4B7749E2 16206306",
    "4B77FFFA 16252921.5",
    "4B821910 17052193",
    "4B868F0E 17636892",
    "4B876520 17746497",
    "4B87D6B8 17804655",
    "4B8958EE 18002395",
    "4B8C3138 18375281",
    "4B8E6E42 18668677",
    "4B90E1D8 1899e4",
    "4B970FE0 198e5",
    "4B998342 20121220",
    "4B99AFB0 20143968",
    "4B9EFAE6 20837836",
    "4BA0EFEE 21094364",
    "4BA17304 21161480",
    "4BA4B683 21589254",
    "4BA5C352 21726885",
    "4BA98AC7 22222222",
    "4BAA977F 22359806",
    "4BAB74BB 22473078",
    "4BAC9FDC 22626233",
    "4BAFF5C0 23063423",
    "4BB03EC4 23100809",
    "4BB550B8 23765361",
    "4BB809B4 24122215",
    "4BBB7268 24569041",
    "4BC360D5 25608618",
    "4BC876A0 26275135",
    "4BC8865D 26283194",
    "4BD04730 27299423",
    "4BD47306 27846156",
    "4BDC7D50 289e5",
    "4BDE68C9 29151634",
    "4BE05D9F 29408062",
    "4BE4E1C0 3e7",
    "4BE6DDA7 30260046",
    "4BE92801 30560258",
    "4BE9EDCF 30661534",
    "4BEDEB96 31184683",
    "4BF36011 31899682",
    "4BF36954 31904423",
    "4BF6409F 32276798",
    "4BF9A486 32721165",
    "4BFD82E8 33228241",
    "4BFD8644 33229960",
    "4BFE502A 33333333",
    "4C00A562 33723784",
    "4C033DE4 34404238",
    "4C051CFE 34894840",
    "4C0583B0 35e6",
    "4C083F9D 35716724",
    "4C09AD09 36090916",
    "4C09C2FF 36113404",
    "4C0A499F 36251260",
    "4C0A8C89 36319780",
    "4C0DDCCB 37188395",
    "4C104B98 37826145",
    "4C111D1D 38040692",
    "4C13092F 38544571",
    "4C131A57 38562139",
    "4C133DE1 38598533",
    "4C140640 38803710",
    "4C1421EA 38832042",
    "4C1793D6 39735126",
    "4C1798D1 39740229",
    "4C188B06 39988247",
    "4C189680 40000000",
    "4C1930F7 40158172",
    "4C19402C 40173744",
    "4C1A22A3 40405643",
    "4C1A49B5 40445652",
    "4C1B0974 40642e3",
    "4C1BA3C0 40800000",
    "4C1C6710 41000000",
    "4C1CB407 41078812",
    "4C1E5D2F 41514172",
    "4C208432 42078409",
    "4C2091AA 42092201",
    "4C21D52C 42423473",
    "4C22C32A 42667174",
    "4C22C502 42669063",
    "4C22F484 42717713",
    "4C2521EE 43288506",
    "4C2554BF 43340539",
    "4C260B66 43527578",
    "4C267A3A 43641062",
    "4C27D8C0 44e6",
    "4C28F9C1 44295939",
    "4C2974CC 44421934",
    "4C298AC7 44444444",
    "4C2BBB93 450187e2",
    "4C2BCC77 45035996.273704985",
    "4C2BCC77 45035996.273704995",
    "4C2EC292 45812296",
    "4C2F0254 45877585",
    "4C2FEBAE 46116538",
    "4C316228 4650e4",
    "4C3241CA 46729e03",
    "4C33B3D5 47107924",
    "4C34C276 47385048",
    "4C353C7C 4751e4",
    "4C357ED7 47577948",
    "4C36E03C 47939822",
    "4C3776F1 48094147",
    "4C37879A 48111209",
    "4C378A28 48113823",
    "4C38100C 48250926",
    "4C387BA6 48361110",
    "4C39852B 48633003",
    "4C39C698 487e5",
    "4C3A43B7 48828125",
    "4C3BA935 49194195",
    "4C3EBC20 50000000",
    "4C3EBC20 5e7",
    "4C3F64A7 50172572",
    "4C42A86B 51028396",
    "4C43B3EF 51302332",
    "4C4ABCB6 53146328",
    "4C4B5762 53304714",
    "4C4DFE60 54e6",
    "4C4FC6F4 54467535",
    "4C502CDC 54571890",
    "4C507F98 54656606",
    "4C50F2F4 54774735",
    "4C51CEF0 55e6",
    "4C521B40 55078143",
    "4C53ED79 55555555",
    "4C546716 55680090",
    "4C559698 55990879",
    "4C559FBD 56000244",
    "4C57B8EA 56550310",
    "4C5A5FF1 57245637",
    "4C5A9928 57304222",
    "4C5C21BE 57706234",
    "4C5C633E 57773302",
    "4C5DE5AC 58169007",
    "4C5DF914 58188878",
    "4C5ED784 58416654",
    "4C5F573E 58547448",
    "4C614E78 59062754",
    "4C632038 59539680",
    "4C644B5C 59846e3",
    "4C64E1C0 60e6",
    "4C64E1C0 6e7",
    "4C689082 60965384",
    "4C6BDF8A 61832742",
    "4C6E06FF 62397437",
    "4C6F6694 62757456",
    "4C6FCEA0 62864002",
    "4C701CD2 62944073",
    "4C705370 63e6",
    "4C70F7D8 63168350",
    "4C72CDD2 63649610",
    "4C737ED7 63830875",
    "4C742400 64e6",
    "4C7574BD 64344821",
    "4C75BDCE 64419642",
    "4C75D374 64441808",
    "4C760E08 64501793",
    "4C77CB28 64957599",
    "4C77F490 650e5",
    "4C786B05 651213e2",
    "4C7C0FBC 66076400",
    "4C7E502A 66666666",
    "4C7E94DA 66737e3",
    "4C805E2A 67301713",
    "4C81E6AE 68105580",
    "4C820CD6 68183731",
    "4C8245D1 68300424",
    "4C82AFB9 68517318",
    "4C835DDE 68873971",
    "4C84774B 69450330",
    "4C84AA79 69555144",
    "4C84EA2D 69685606",
    "4C84EAB0 69686659",
    "4C8500B2 69731732",
    "4C856E45 69956135",
    "4C8583B0 70e6",
    "4C867FF6 70516656",
    "4C875EB4 70972830",
    "4C876BF8 71e6",
    "4C881818 71352512",
    "4C8A0C4B 72376922",
    "4C8B06DC 72890080",
    "4C8B8DEE 73166703",
    "4C8C1066 73433906",
    "4C8D47E6 74071854",
    "4C8D53A6 74095920",
    "4C8DE669 74396490",
    "4C8F7E94 75232413",
    "4C9000C1 75499016",
    "4C9086E4 75773725",
    "4C909486 75801648",
    "4C916E5A 76247763",
    "4C91B8B0 76400000",
    "4C924EDB 76707543",
    "4C92560D 76722281",
    "4C9266CE 76756590",
    "4C931780 77118461",
    "4C93E007 77529147",
    "4C94596E 77777777",
    "4C96C2DE 79042286",
    "4C977803 79413272",
    "4C983B64 79813406",
    "4C98404F 79823479",
    "4C989680 80000000",
    "4C989680 80e6",
    "4C989680 8e7",
    "4C9902F5 80222122",
    "4C9C7003 82018330",
    "4C9DD0C9 82740809",
    "4C9DD3A0 82746620",
    "4C9E0069 82838342",
    "4C9EC178 83233730",
    "4C9FAA89 83711047",
    "4C9FB1A1 83725573",
    "4C9FC9F8 83775423",
    "4CA049E9 84037448",
    "4CA15A3B 84595161",
    "4CA2494E 85084788",
    "4CA2C726 85342511",
    "4CA39B68 85777219",
    "4CA39F84 85785631",
    "4CA3D982 85904404",
    "4CA434D9 86091465",
    "4CA46494 86189216",
    "4CA4F4BB 86484437",
    "4CA583B6 86777268",
    "4CA5DE2A 86962508",
    "4CA75842 87736852",
    "4CA772C7 87791158",
    "4CA79837 87867832",
    "4CA91A86 88658996",
    "4CA98AC7 88888888",
    "4CA9AE6E 88961903",
    "4CAA0527 89139509",
    "4CAAF463 89629465",
    "4CAB2E71 89748360",
    "4CABAFA0 90012929",
    "4CAC5B2C 90364256",
    "4CAD8168 90966848",
    "4CAE418F 91360377",
    "4CAE5E60 91419389",
    "4CAF8C7D 92038121",
    "4CAFB717 92125366",
    "4CAFEEC6 92239407",
    "4CAFFC4E 92267121",
    "4CB2023C 93327838",
    "4CB43063 94470938",
    "4CB54CCD 95053418",
    "4CB55FC0 95092224",
    "4CB59EA6 95221044",
    "4CB6083B 95437272",
    "4CB6C1FD 95817703",
    "4CB811DB 96505557",
    "4CB873C8 96706114",
    "4CB892F4 96769950",
    "4CB8CA18 96882881",
    "4CB8D1AB 96898389",
    "4CB97520 97233152",
    "4CBA389F 97633526",
    "4CBB448C 98182244",
    "4CBBE9B3 98520472",
    "4CBCBE20 98955522",
    "4CBCC438 98968000",
    "4CBD2764 99171105",
    "4CBD7B8E 99343471",
    "4CBD9728 994e5",
    "4CBEBC20 99999999",
    "4CBEBC20 100000000",
    "4CBEBC20 10e7",
    "4CBEBC20 1e8",
    "4CD3ED79 111111111",
    "4CEB79A3 123456789",
    "4CF1776B 126597973",
    "4D06794E 141006042",
    "4D0B1EA5 145877585",
    "4D0B5A4E 146121952",
    "4D4EB950 216765690",
    "4D53ED79 222222222",
    "4D68D4A5 244140625",
    "4D81EC22 272467e3",
    "4D8F0D18 3e8",
    "4D95E1D4 314325637",
    "4D9EF21B 333333333",
    "4DB56464 380406926",
    "4DBEBC20 4e8",
    "4DD3ED79 444444444",
    "4DE4E1C0 480000000",
    "4DEE6B28 500000000",
    "4E04746C 555555555",
    "4E0CD6EE 590723948",
    "4E0CE5C1 590966848",
    "4E0E440D 596706114",
    "4E104361 605083704",
    "4E15801D 627050305",
    "4E16FD29 633293366",
    "4E1EF21B 666666666",
    "4E26E49C 7e8",
    "4E2B2F3E 718e6",
    "4E2BA950 72e7",
    "4E2DC893 728900802",
    "4E310CB5 7426e5",
    "4E344939 756174393",
    "4E354C05 760414536",
    "4E396FCA 777777777",
    "4E3EBC20 800000000",
    "4E3F43B3 802221226",
    "4E401940 805720085",
    "4E465D40 832e6",
    "4E4A4690 848405530",
    "4E4C2A88 856334878",
    "4E4F8C1F 870516656",
    "4E517E45 878678326",
    "4E53ED79 888888888",
    "4E5479AC 891185938",
    "4E55F9B9 897478238",
    "4E5693A4 9e08",
    "4E5693A4 9e8",
    "4E57C9B2 905079926",
    "4E58F5FE 91e07",
    "4E59B814 913179899",
    "4E6313B1 952429603",
    "4E64E1C0 96e7",
    "4E69A674 98e7",
    "4E6B6C3C 987434744",
    "4E6C713D 991711052",
    "4E6E6B28 999999999",
    "4E6E6B28 1e9",
    "4E6E6B29 1.00000006e+09",
    "4E9184E7 1220703125",
    "4E932C06 1234567890",
    "4EEE6B28 2e09",
    "4EFFFFFD 2147483314",
    "4EFFFFFD 2147483315",
    "4EFFFFFE 2147483351",
    "4EFFFFFE 2147483352",
    "4EFFFFFE 2147483388",
    "4EFFFFFE 2147483389",
    "4EFFFFFE 2147483425",
    "4EFFFFFE 2147483426",
    "4EFFFFFF 2147483462",
    "4EFFFFFF 2147483463",
    "4EFFFFFF 2147483499",
    "4EFFFFFF 2147483500",
    "4EFFFFFF 2147483536",
    "4EFFFFFF 2147483537",
    "4EFFFFFF 2147483573",
    "4EFFFFFF 2147483574",
    "4F000000 2147483610",
    "4F000000 2147483611",
    "4F000000 2147483647",
    "4F000000 2147483648",
    "4F1502F9 25e8",
    "4F18542C 2555653131",
    "4F26C8DF 2798182244",
    "4F3EBD30 3200069671",
    "4F41562C 3243650005",
    "4F44EA11 3303674053",
    "4F52DEF1 3537826145",
    "4F6E6B28 4e9",
    "4F712632 4045812296",
    "4F820D2F 4363804324",
    "4F920719 4899877186",
    "4F9502F9 5e09",
    "4F9502F9 5e9",
    "4F9528FA 5004981478",
    "4F9796DB 5086492111",
    "4F9923CB 5138519684",
    "4FA01870 5371912364",
    "4FA7F185 5635246428",
    "4FAB42E9 5746577930",
    "4FB5E621 6103515625",
    "4FBBC130 63e8",
    "4FD09DC3 70e8",
    "4FDC0D5B 7383725573",
    "4FE3C54F 7642717713",
    "4FEBBC83 791e07",
    "4FEE6B28 80e8",
    "4FEE6B28 8e9",
    "5000B5F0 8637627989",
    "50041CB9 8865899617",
    "5005BC48 8974836059",
    "50061C46 9e9",
    "50097D1D 9226712162",
    "500C1228 94e8",
    "500C600D 942042e4",
    "501022C7 9672793580",
    "50119F04 9772470297",
    "501502F9 1e10",
    "5041B710 1.3e10",
    "5077A845 1662e7",
    "50B7BD75 24661173473",
    "50CDA2D2 276e8",
    "50D59BAF 2867e7",
    "50E35FA9 30517578125",
    "50E6F7CF 31e9",
    "50F52E46 32907604691",
    "51093AA0 36837130890",
    "511502F9 40000000000",
    "51197B62 412e08",
    "51325DC2 47879823479",
    "5139F1C6 49914078536",
    "513DFD64 51000000000",
    "51492A6A 54e9",
    "51559F80 573440e5",
    "515BCAC9 59e9",
    "51861C46 72E9",
    "5189D5F3 74000000000",
    "519A997C 83e9",
    "51A7A358 90000000000",
    "51BA43B7 100000000000",
    "51BA43B7 1e11",
    "51BC208E 101e9",
    "520840A3 1463e8",
    "520E1BCA 152587890625",
    "523A43B7 200000000000",
    "5261D86F 242499697392",
    "5282629A 280000000000",
    "528BB2C9 3e11",
    "52BA43B7 400000000000",
    "52FAFDCF 539e9",
    "530E81C6 612062576589",
    "530EA44D 612641865679",
    "5330644C 757596946075",
    "5331A2BC 762939453125",
    "533EEB57 819992132456",
    "5368D4A5 1e12",
    "53E8D4A5 2e12",
    "542E9F7C 3e12",
    "545E0B6B 3814697265625",
    "549184E7 5e012",
    "549BD269 5354e9",
    "54AC2BC4 59157491e5",
    "54CD5F7D 7056562757456",
    "55024FFA 8955e9",
    "550715AE 92829494e5",
    "550C4B86 9641e9",
    "551184E7 10000000000000",
    "551184E7 1e13",
    "558AC723 19073486328125",
    "559184E7 20000000000000",
    "55CBBA10 28000000000000",
    "561184E7 40000000000000",
    "561C4000 42949672960001",
    "5699E0B1 84595161401484",
    "56A2D793 89523386091465",
    "56AD78EC 95367431640625",
    "56B5E621 1e14",
    "5783EEC3 290123e9",
    "579F295D 350000000000000",
    "57B5E621 400000000000000",
    "57B5E621 40e13",
    "57C0D019 424000000000000",
    "57D4DED7 468107100525890",
    "57D8D727 476837158203125",
    "57E35FA9 5e14",
    "5809CD5F 606060606060606",
    "58267002 732000000000000",
    "5835E621 800000000000000",
    "5835E621 8e14",
    "58635FA9 1000000000000000",
    "58635FA9 1e15",
    "58800000 1125899906842624.125",
    "58800000 1125899906842901.875",
    "58956F79 1314448000000000",
    "59000000 2251799813685248.25",
    "59000000 2251799813685803.75",
    "59078678 2384185791015625",
    "593D7A62 3333333333333333",
    "59635FA9 4000000000000000",
    "59800000 4503599627370496.5",
    "59800000 4503599627370497.5",
    "59800000 4503599627475352.5",
    "59800000 4503599627475353.5",
    "598E1BCA 5000000000000000",
    "59D529AF 75e14",
    "59E35FA9 8000000000000000",
    "5A000000 9007199254740992",
    "5A000000 9007199254740993",
    "5A000000 9007199254740994",
    "5A000000 9007199254740995",
    "5A09D865 97e14",
    "5A0CA459 9896800000000000",
    "5A0E1BCA 1e16",
    "5A296816 11920928955078125",
    "5B53C21C 59604644775390625",
    "5B64158F 642e14",
    "5BB1A2BC 1e17",
    "5C845951 298023223876953125",
    "5CCE8061 465e15",
    "5D5E0B6B 1e18",
    "5DA56FA6 1490116119384765625",
    "5E056279 2402844368454405395.2",
    "5ECECB8F 7450580596923828125",
    "5EF9CCD9 9e18",
    "5F053A0D 96e017",
    "5F08DFC5 9.862818194192001e18",
    "5F0AC723 1e19",
    "60805E9A 74e18",
    "609C2007 9e19",
    "60A3959D 943e17",
    "60AD78EC 1e20",
    "6258D727 1e21",
    "64078678 1e22",
    "657BBCFA 743e20",
    "659CFBB2 92666518056446206563E3",
    "65A96816 1e23",
    "65A96816 9.999999999999999e22",
    "6723E072 773886e18",
    "6753C21C 1e24",
    "6C186875 737e24",
    "6D1B18AB 3e27",
    "715C3BD0 1090544144181609348835077142190",
    "717A6A7F 124e28",
    "729C8290 62e29",
    "73FC6F7C 4e31",
    "745642CC 67902e27",
    "7487AF20 86e30",
    "749DC5AE 1e32",
    "760A0CF8 7e32",
    "771209B2 29620e29",
    "777684DF 5e33",
    "791A130C 5e34",
    "7E967699 1e38",
    "7F000000 1.7014118346046923e+38",
    "7F7FFFFF 3.4028234664e38",
    "7F7FFFFF 3.4028234665e38",
    "7F7FFFFF 3.4028234666e38",
    "7F800000 3.5028234666e38",
    "7F800000 51823e34",
    "7F800000 555e36",
    "7F800000 912e37",
    "7F800000 92487298e33",
    "7F800000 778e39",
    "7F800000 8e41",
    "7F800000 81e40",
    "7F800000 90e40",
    "7F800000 1.7339253062092163730578609458683877051596800000000000000000000000e+42",
    "7F800000 1778e39",
    "7F800000 598e40",
    "7F800000 808e40",
    "7F800000 203e41",
    "7F800000 89e42",
    "7F800000 2e44",
    "7F800000 5e44",
    "7F800000 2.1470977154320536489471030463761883783915110400000000000000000000e+45",
    "7F800000 6e45",
    "7F800000 2091e44",
    "7F800000 368e45",
    "7F800000 967e45",
    "7F800000 442e46",
    "7F800000 7442e45",
    "7F800000 4e49",
    "7F800000 892091e44",
    "7F800000 5e50",
    "7F800000 7e50",
    "7F800000 2e51",
    "7F800000 3e51",
    "7F800000 39e50",
    "7F800000 7e51",
    "7F800000 1.9189205311132686907264385602245237137907390376574976000000000000e+52",
    "7F800000 2.0972622234386619214559824785284023792871122537545728000000000000e+52",
    "7F800000 1e53",
    "7F800000 1.7664960224650106892054063261344555646357024359107788800000000000e+53",
    "7F800000 335e51",
    "7F800000 2e54",
    "7F800000 2.8184483231688951563253238886553506793085187889855201280000000000e+54",
    "7F800000 6472e51",
    "7F800000 2e55",
    "7F800000 5e55",
    "7F800000 5667844e49",
    "7F800000 62e055",
    "7F800000 8e056",
    "7F800000 1.0001803374372191849407179462120053338028379051879898808320000000e+57",
    "7F800000 5e57",
    "7F800000 1.8607245283054342363818436991534856973992070520151142825984000000e+58",
    "7F800000 7.0420557077594588669468784357561207962098443483187940792729600000e+59",
    "7F800000 9e59",
    "7F800000 99e59",
    "7F800000 13e60",
    "7F800000 4.4900312744003159009338275160799498340862630046359789166919680000e+61",
    "7F800000 1e64",
    "7F800000 88e65",
    "7F800000 3e69",
    "7F800000 59e68",
    "7F800000 86e69",
    "7F800000 2224e68",
    "7F800000 9e74",
    "7F800000 42e74",
    "7F800000 7.2370055773322621e+75",
    "7F800000 9184e72",
    "7F800000 93e74",
    "7F800000 40041e073",
    "7F800000 7e77",
    "7F800000 44e80",
    "7F800000 9e82",
    "7F800000 242e81",
    "7F800000 527e81",
    "7F800000 9e83",
    "7F800000 9e84",
    "7F800000 1e85",
    "7F800000 7e85",
    "7F800000 1e86",
    "7F800000 503e085",
    "7F800000 112e86",
    "7F800000 6e88",
    "7F800000 5e89",
    "7F800000 8358109e84",
    "7F800000 90054602635948575728E72",
    "7F800000 9271e88",
    "7F800000 9153e89",
    "7F800000 4e093",
    "7F800000 3e95",
    "7F800000 3608e92",
    "7F800000 791e093",
    "7F800000 8545e94",
    "7F800000 80e98",
    "7F800000 980e98",
    "7F800000 83126e97",
    "7F800000 7e105",
    "7F800000 538e122",
    "7F800000 7e127",
    "7F800000 1e128",
    "7F800000 7795e136",
    "7F800000 7549530e188",
    "7F800000 71e223",
    "7F800000 4e250",
    "7F800000 1e256",
    "7F800000 1.01e256",
    "7F800000 67902e276",
    "7F800000 9748e282",
    "7F800000 1e300",
    "7F800000 19e306",
    "7F800000 1.797693134862315700000000000000001e308",
    "7F800000 1.7976931348623157e308",
    "7F800000 1.7976931348623158e308",
    "7F800000 0.1e310",
    "7F800000 1.832312213213213232132132143451234453123412321321312e308",
    "7F800000 1.8e308",
    "7F800000 1.9e308",
    "7F800000 1234456789012345678901234567890e9999999999999999999999999999",
    "7F800000 12e1342",
    "7F800000 12e13424",
    "7F800000 148e3032",
    "7F800000 18e88640",
    "7F800000 1e1000",
    "7F800000 1e1853",
    "7F800000 1e414218",
    "7F800000 2139879401095466344511101915470454744.9813888656856943E+272",
    "7F800000 21e440",
    "7F800000 21e44003",
    "7F800000 246723473e3813",
    "7F800000 24e4421730",
    "7F800000 2529e2734",
    "7F800000 26e077774",
    "7F800000 28e2557",
    "7F800000 2e08987",
    "7F800000 2e0898765",
    "7F800000 2e3000",
    "7F800000 2e30000000000000000",
    "7F800000 2e69915",
    "7F800000 2e801",
    "7F800000 322e62600000",
    "7F800000 35e702",
    "7F800000 39e436",
    "7F800000 3e60868",
    "7F800000 3e84959",
    "7F800000 3e8495912",
    "7F800000 412e0806",
    "7F800000 4196e952",
    "7F800000 43e40076",
    "7F800000 442e4688",
    "7F800000 44e864",
    "7F800000 464e3945",
    "7F800000 473e3813",
    "7F800000 47859e743",
    "7F800000 486e494",
    "7F800000 49e807",
    "7F800000 50e4395",
    "7F800000 51e1566",
    "7F800000 5400e987",
    "7F800000 549e57273",
    "7F800000 555e361951",
    "7F800000 5e330",
    "7F800000 5e5728",
    "7F800000 5e620",
    "7F800000 5e7873",
    "7F800000 5e823",
    "7F800000 5e82392",
    "7F800000 626e974",
    "7F800000 62e2929",
    "7F800000 6472e511",
    "7F800000 696e840",
    "7F800000 6e188853",
    "7F800000 6e356932",
    "7F800000 6e3569326",
    "7F800000 6e804",
    "7F800000 6e984",
    "7F800000 71758e219652",
    "7F800000 718e68396",
    "7F800000 71e914",
    "7F800000 71e91432",
    "7F800000 7230489e80000",
    "7F800000 74e608",
    "7F800000 75e2224",
    "7F800000 7859e743",
    "7F800000 7868e94050",
    "7F800000 788035e61382",
    "7F800000 79e1632",
    "7F800000 7e05401",
    "7F800000 7e12780",
    "7F800000 7e2000",
    "7F800000 7e334",
    "7F800000 7e33455637",
    "7F800000 7e41392",
    "7F800000 7e670471",
    "7F800000 7e67047175",
    "7F800000 7e998",
    "7F800000 808e1755",
    "7F800000 808e17555",
    "7F800000 83126e978",
    "7F800000 83e872",
    "7F800000 8955e946",
    "7F800000 89e80000",
    "7F800000 8e1456105",
    "7F800000 8e43145",
    "7F800000 8e431456",
    "7F800000 8e5087",
    "7F800000 8e5410",
    "7F800000 8e5410288",
    "7F800000 8e679",
    "7F800000 8e88640",
    "7F800000 8e905",
    "7F800000 8e921",
    "7F800000 8e938662",
    "7F800000 8e938662882",
    "7F800000 8e952",
    "7F800000 915e2486",
    "7F800000 945e455",
    "7F800000 953e862",
    "7F800000 953e8624",
    "7F800000 963e6685",
    "7F800000 963e66858",
    "7F800000 964e858",
    "7F800000 96e952",
    "7F800000 9748e2826",
    "7F800000 976e4108",
    "7F800000 976e41088617",
    "7F800000 9837e699095",
    "7F800000 98e94712",
    "7F800000 98e947129",
    "7F800000 99e59958885",
    "7F800000 99e619",
    "7F800000 9e1632",
    "7F800000 9e20735",
    "7F800000 9e40000000",
    "7F800000 9e563",
    "7F800000 9e795",
    "7F800000 9e904"
)
