/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package androidx.health.connect.client.request

import androidx.health.connect.client.aggregate.AggregateMetric
import androidx.health.connect.client.records.metadata.DataOrigin
import androidx.health.connect.client.time.TimeRangeFilter
import java.time.Duration

/**
 * Request object to read time bucketed aggregations for given [AggregateMetric]s in Android Health
 * Platform.
 *
 * [timeRangeSlicer] contains a [Duration] of fixed physical time intervals, such as per hour, per
 * ten minutes or so. Prefer [AggregateGroupByPeriodRequest], if you would like variable length time
 * intervals, such as per day, which may or may not include DST (23 or 25 hour).
 *
 * @param metrics Set of [AggregateMetric]s to aggregate.
 * @param timeRangeFilter The [TimeRangeFilter] to read from.
 * @param timeRangeSlicer The bucket size of each returned aggregate row. [timeRangeFilter] will be
 * sliced into several equal-sized time buckets (except for the last one).
 * @param dataOriginFilter Set of [DataOrigin]s to read from, or empty for no filter.
 */
class AggregateGroupByDurationRequest(
    internal val metrics: Set<AggregateMetric<*>>,
    internal val timeRangeFilter: TimeRangeFilter,
    internal val timeRangeSlicer: Duration,
    internal val dataOriginFilter: Set<DataOrigin> = emptySet(),
) {
    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as AggregateGroupByDurationRequest

        if (metrics != other.metrics) return false
        if (timeRangeFilter != other.timeRangeFilter) return false
        if (timeRangeSlicer != other.timeRangeSlicer) return false
        if (dataOriginFilter != other.dataOriginFilter) return false

        return true
    }

    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun hashCode(): Int {
        var result = metrics.hashCode()
        result = 31 * result + timeRangeFilter.hashCode()
        result = 31 * result + timeRangeSlicer.hashCode()
        result = 31 * result + dataOriginFilter.hashCode()
        return result
    }
}
