/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.car.drawer;

import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.car.R;
import androidx.car.util.CarUxRestrictionsUtils;
import androidx.car.uxrestrictions.CarUxRestrictions;
import androidx.car.uxrestrictions.OnUxRestrictionsChangedListener;
import androidx.recyclerview.widget.RecyclerView;

/**
 * Re-usable {@link RecyclerView.ViewHolder} for displaying items in the
 * {@link androidx.car.drawer.CarDrawerAdapter}.
 */
public class DrawerItemViewHolder extends RecyclerView.ViewHolder implements
        OnUxRestrictionsChangedListener {
    private final ImageView mIcon;
    private final TextView mTitle;
    private final TextView mBody;
    private final ImageView mEndIcon;

    /**
     * Create a new {@code ViewHolder} that will parse the given view that is meant to displaying
     * items in a drawer.
     *
     * <p>The given view is required to have a title and icon view with an optional body and end
     * icon view. These views are looked up via ids, specifically:
     *
     * <ul>
     *     <li>Icon - R.id.icon
     *     <li>Title - R.id.title
     *     <li>Body - R.id.body
     *     <li>End Icon - R.id.end_icon
     * </ul>
     *
     * <p>This method is exposed for testing purposes. The {@code DrawerItemViewHolder} is
     * automatically created by the {@link CarDrawerAdapter}.
     *
     * @param view The view that will represent an item in a drawer.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PACKAGE_PRIVATE)
    public DrawerItemViewHolder(@NonNull View view) {
        super(view);
        mIcon = view.findViewById(R.id.icon);
        if (mIcon == null) {
            throw new IllegalArgumentException("Icon view cannot be null!");
        }

        mTitle = view.findViewById(R.id.title);
        if (mTitle == null) {
            throw new IllegalArgumentException("Title view cannot be null!");
        }

        // Next two are optional and may be null.
        mBody = view.findViewById(R.id.body);
        mEndIcon = view.findViewById(R.id.end_icon);
    }

    /** Returns the view that should be used to display the main icon. */
    @NonNull
    public ImageView getIconView() {
        return mIcon;
    }

    /** Returns the view that will display the main title. */
    @NonNull
    public TextView getTitleView() {
        return mTitle;
    }

    /**
     * Returns the view that is used for the body text that is smaller than the title text and
     * appears beneath the title.
     */
    @Nullable
    public TextView getBodyView() {
        return mBody;
    }

    /** Returns the icon that is displayed at the end of the view. */
    @Nullable
    public ImageView getEndIconView() {
        return mEndIcon;
    }

    /**
     * Updates child views with current car UX restrictions.
     *
     * <p>{@code Text} might be truncated to meet length limit required by regulation.
     *
     * @param restrictionsInfo current car UX restrictions.
     */
    @Override
    public void onUxRestrictionsChanged(CarUxRestrictions restrictionsInfo) {
        CarUxRestrictionsUtils.apply(itemView.getContext(), restrictionsInfo, getBodyView());
    }
}
