/*
 * Copyright 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.ui.test.util

import androidx.ui.core.Duration
import androidx.ui.core.IntPxSize
import androidx.ui.core.PointerEventPass
import androidx.ui.core.PointerInputChange
import androidx.ui.core.PointerInputData
import androidx.ui.core.gesture.util.VelocityTracker
import androidx.ui.test.util.PointerInputRecorder.DataPoint
import com.google.common.truth.Truth.assertThat

class PointerInputRecorder {

    data class DataPoint(val id: Int, val data: PointerInputData) {
        val timestamp get() = data.uptime!!
        val position get() = data.position
        val x get() = data.position!!.x
        val y get() = data.position!!.y
        val down get() = data.down
    }

    private val _events = mutableListOf<DataPoint>()
    val events get() = _events as List<DataPoint>

    private val velocityTracker = VelocityTracker()
    val recordedVelocity get() = velocityTracker.calculateVelocity()

    fun onPointerInput(
        changes: List<PointerInputChange>,
        pass: PointerEventPass,
        @Suppress("UNUSED_PARAMETER") bounds: IntPxSize
    ): List<PointerInputChange> {
        if (pass == PointerEventPass.InitialDown) {
            changes.forEach {
                _events.add(DataPoint(it.id, it.current))
                velocityTracker.addPosition(it.current.uptime!!, it.current.position!!)
            }
        }
        return changes
    }
}

val PointerInputRecorder.downEvents get() = events.filter { it.down }

val PointerInputRecorder.recordedDuration: Duration
    get() {
        check(events.isNotEmpty()) { "No events recorded" }
        return events.last().timestamp - events.first().timestamp
    }

fun PointerInputRecorder.assertTimestampsAreIncreasing() {
    check(events.isNotEmpty()) { "No events recorded" }
    events.reduce { prev, curr ->
        assertThat(curr.timestamp).isAtLeast(prev.timestamp)
        curr
    }
}

fun PointerInputRecorder.assertOnlyLastEventIsUp() {
    check(events.isNotEmpty()) { "No events recorded" }
    assertThat(events.last().down).isFalse()
    assertThat(events.count { !it.down }).isEqualTo(1)
}

/**
 * Checks that the coordinates are progressing in a monotonous direction
 */
fun List<DataPoint>.isMonotonousBetween(x0: Float, y0: Float, x1: Float, y1: Float) {
    map { it.x.value }.isMonotonousBetween(x0, x1, 1e-3f)
    map { it.y.value }.isMonotonousBetween(y0, y1, 1e-3f)
}
