/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.wear.complications;

import static com.google.common.truth.Truth.assertThat;

import static org.junit.Assert.assertNull;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;

import android.content.Intent;
import android.os.RemoteException;
import android.support.wearable.complications.IComplicationManager;
import android.support.wearable.complications.IComplicationProvider;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.wear.complications.data.ComplicationData;
import androidx.wear.complications.data.ComplicationType;
import androidx.wear.complications.data.LongTextComplicationData;
import androidx.wear.complications.data.PlainComplicationText;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.robolectric.annotation.internal.DoNotInstrument;
import org.robolectric.shadows.ShadowLooper;

/** Tests for {@link ComplicationProviderService}. */
@RunWith(ComplicationsTestRunner.class)
@DoNotInstrument
public class ComplicationProviderServiceTest {

    private static final String TAG = "ComplicationProviderServiceTest";

    @Mock
    private IComplicationManager mRemoteManager;
    private IComplicationManager.Stub mLocalManager = new IComplicationManager.Stub() {
        @Override
        public void updateComplicationData(int complicationId,
                android.support.wearable.complications.ComplicationData data)
                throws RemoteException {
            mRemoteManager.updateComplicationData(complicationId, data);
        }
    };

    private IComplicationProvider.Stub mComplicationProvider;
    private IComplicationProvider.Stub mNoUpdateComplicationProvider;

    private ComplicationProviderService mTestService = new ComplicationProviderService() {

        @Override
        public void onComplicationUpdate(
                int complicationId,
                @NonNull ComplicationType type,
                @NonNull ComplicationUpdateCallback callback) {
            try {
                callback.onUpdateComplication(
                        new LongTextComplicationData.Builder(
                                new PlainComplicationText.Builder("hello " + complicationId).build()
                        ).build()
                );
            } catch (RemoteException e) {
                Log.e(TAG, "onComplicationUpdate failed with error: ", e);
            }
        }

        @Nullable
        @Override
        public ComplicationData getPreviewData(@NonNull ComplicationType type) {
            if (type == ComplicationType.PHOTO_IMAGE) {
                return null;
            }
            return new LongTextComplicationData.Builder(
                    new PlainComplicationText.Builder("hello preview").build()
            ).build();
        }
    };

    private ComplicationProviderService mNoUpdateTestService = new ComplicationProviderService() {

        @Override
        public void onComplicationUpdate(
                int complicationId,
                @NonNull ComplicationType type,
                @NonNull ComplicationUpdateCallback callback) {
            try {
                // Null means no update required.
                callback.onUpdateComplication(null);
            } catch (RemoteException e) {
                Log.e(TAG, "onComplicationUpdate failed with error: ", e);
            }
        }

        @Nullable
        @Override
        public ComplicationData getPreviewData(@NonNull ComplicationType type) {
            return new LongTextComplicationData.Builder(
                    new PlainComplicationText.Builder("hello preview").build()
            ).build();
        }
    };

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        mComplicationProvider =
                (IComplicationProvider.Stub) mTestService.onBind(
                        new Intent(ComplicationProviderService.ACTION_COMPLICATION_UPDATE_REQUEST));

        mNoUpdateComplicationProvider =
                (IComplicationProvider.Stub) mNoUpdateTestService.onBind(
                        new Intent(ComplicationProviderService.ACTION_COMPLICATION_UPDATE_REQUEST));
    }

    @Test
    public void testOnComplicationUpdate() throws Exception {
        int id = 123;
        mComplicationProvider.onUpdate(
                id, ComplicationType.LONG_TEXT.asWireComplicationType(), mLocalManager);
        ShadowLooper.runUiThreadTasks();

        ArgumentCaptor<android.support.wearable.complications.ComplicationData> data =
                ArgumentCaptor.forClass(
                        android.support.wearable.complications.ComplicationData.class);
        verify(mRemoteManager).updateComplicationData(eq(id), data.capture());
        assertThat(data.getValue().getLongText().getTextAt(null, 0)).isEqualTo(
                "hello " + id
        );
    }

    @Test
    public void testOnComplicationUpdateNoUpdateRequired() throws Exception {
        int id = 123;
        mNoUpdateComplicationProvider.onUpdate(
                id, ComplicationType.LONG_TEXT.asWireComplicationType(), mLocalManager);
        ShadowLooper.runUiThreadTasks();

        ArgumentCaptor<android.support.wearable.complications.ComplicationData> data =
                ArgumentCaptor.forClass(
                        android.support.wearable.complications.ComplicationData.class);
        verify(mRemoteManager).updateComplicationData(eq(id), data.capture());
        assertThat(data.getValue()).isNull();
    }

    @Test
    public void testGetComplicationPreviewData() throws Exception {
        assertThat(mComplicationProvider.getComplicationPreviewData(
                ComplicationType.LONG_TEXT.asWireComplicationType()
        ).getLongText().getTextAt(null, 0)).isEqualTo("hello preview");
    }

    @Test
    public void testGetComplicationPreviewDataReturnsNull() throws Exception {
        // The ComplicationProvider doesn't support PHOTO_IMAGE so null should be returned.
        assertNull(mComplicationProvider.getComplicationPreviewData(
                ComplicationType.PHOTO_IMAGE.asWireComplicationType())
        );
    }
}
