/*
 * Copyright 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.appactions.interaction.capabilities.core

import java.util.Objects
/**
 * Class that represents the response after a Capability fulfills an action.
 */
class ExecutionResult<OutputT> internal constructor(
    @get:JvmName("shouldStartDictation")
    val shouldStartDictation: Boolean,
    val output: OutputT?,
) {
    override fun toString() =
        "ExecutionResult(shouldStartDictation=$shouldStartDictation,output=$output)"

    override fun equals(other: Any?): Boolean {
        return other is ExecutionResult<*> && output == other.output
    }

    override fun hashCode() = Objects.hash(shouldStartDictation, output)

    /**
     * Builder for ExecutionResult.
     */
    class Builder<OutputT> {
        private var shouldStartDictation: Boolean = false
        private var output: OutputT? = null

        /** Sets whether or not this fulfillment should start dictation. */
        fun setStartDictation(startDictation: Boolean) = apply {
            this.shouldStartDictation = startDictation
        }

        /** Sets the execution output. */
        fun setOutput(output: OutputT) = apply {
            this.output = output
        }

        /** Builds and returns the ExecutionResult instance. */
        fun build() = ExecutionResult(shouldStartDictation, output)
    }
}
