/*
 * Copyright 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.inspection.framework

import android.view.View
import android.view.ViewGroup
import androidx.compose.ui.inspection.util.ThreadUtils

fun ViewGroup.getChildren(): List<View> {
    ThreadUtils.assertOnMainThread()
    return (0 until childCount).map { i -> getChildAt(i) }
}

fun View.ancestors(): Sequence<View> =
    generateSequence(this) { it.parent as? View }

fun View.isRoot(): Boolean =
    parent as? View == null

/**
 * Return a list of this view and all its children in depth-first order
 */
fun View.flatten(): Sequence<View> {
    ThreadUtils.assertOnMainThread()

    val remaining = mutableListOf(this)
    return generateSequence {
        val next = remaining.removeLastOrNull()
        if (next is ViewGroup) {
            remaining.addAll(next.getChildren())
        }
        next
    }
}
