/*
 * Copyright 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.ui.node

import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.layout.boundsInRoot
import androidx.compose.ui.semantics.SemanticsActions
import androidx.compose.ui.semantics.SemanticsConfiguration
import androidx.compose.ui.semantics.getOrNull

/**
 * A [Modifier.Node] that adds semantics key/value for use in testing,
 * accessibility, and similar use cases.
 *
 * This is the [androidx.compose.ui.Modifier.Node] equivalent of
 * [androidx.compose.ui.semantics.SemanticsModifier]
 */
interface SemanticsModifierNode : DelegatableNode {
    /**
     * The SemanticsConfiguration holds substantive data, especially a list of key/value pairs
     * such as (label -> "buttonName").
     */
    val semanticsConfiguration: SemanticsConfiguration
}

fun SemanticsModifierNode.invalidateSemantics() = requireOwner().onSemanticsChange()

internal val SemanticsModifierNode.useMinimumTouchTarget: Boolean
    get() = semanticsConfiguration.getOrNull(SemanticsActions.OnClick) != null

internal val SemanticsConfiguration.useMinimumTouchTarget: Boolean
    get() = getOrNull(SemanticsActions.OnClick) != null

internal fun SemanticsModifierNode.touchBoundsInRoot(): Rect {
    if (!node.isAttached) {
        return Rect.Zero
    }
    if (!useMinimumTouchTarget) {
        return requireCoordinator(Nodes.Semantics).boundsInRoot()
    }

    return requireCoordinator(Nodes.Semantics).touchBoundsInRoot()
}

internal fun Modifier.Node.touchBoundsInRoot(useMinimumTouchTarget: Boolean): Rect {
    if (!node.isAttached) {
        return Rect.Zero
    }
    if (!useMinimumTouchTarget) {
        return requireCoordinator(Nodes.Semantics).boundsInRoot()
    }

    return requireCoordinator(Nodes.Semantics).touchBoundsInRoot()
}