## Android Lint Guidelines

### Suppression vs Baselines

Lint sometimes flags false positives, even though it is safe to ignore these
errors (for example `WeakerAccess` warnings when you are avoiding synthetic
access). There may also be lint failures when your library is in the middle of a
beta / rc / stable release, and cannot make the breaking changes needed to fix
the root cause. There are two ways of ignoring lint errors:

1.  Suppression - using `@SuppressLint` (for Java) or `@Suppress` annotations to
    ignore the warning per call site, per method, or per file.

    Note: `@SuppressLint` requires Android dependency

2.  Baselines - allowlisting errors in a `lint-baseline.xml` file at the root of
    the project directory.

Where possible, you should use a **suppression annotation at the call site**.
This helps ensure that you are only suppressing the *exact* failure, and this
also keeps the failure visible so it can be fixed later on. Only use a baseline
if you are in a Java library without Android dependencies, or when enabling a
new lint check, and it is prohibitively expensive / not possible to fix the
errors generated by enabling this lint check.

To update a lint baseline after you have fixed issues, run the
`updateLintBaseline` task.

```shell
./gradlew :core:core:updateLintBaseline
```

## Metalava API Lint

As well as Android Lint, which runs on all source code, Metalava will also run
checks on the public API surface of each library. Similar to with Android Lint,
there can sometimes be false positives / intended deviations from the API
guidelines that Metalava will lint your API surface against. When this happens,
you can suppress Metalava API lint issues using `@SuppressLint` (for Java) or
`@Suppress` annotations. In cases where it is not possible, update Metalava's
baseline with the `updateApiLintBaseline` task.

```shell
./gradlew :core:core:updateApiLintBaseline
```

This will create/amend the `api_lint.ignore` file that lives in a library's
`api` directory.

## Build Output Guidelines

In order to more easily identify the root cause of build failures, we want to
keep the amount of output generated by a successful build to a minimum.
Consequently, we track build output similarly to the way in which we track Lint
warnings.

### Invoking build output validation

You can add `-Pandroidx.validateNoUnrecognizedMessages` to any other AndroidX
gradlew command to enable validation of build output. For example:

```shell
/gradlew -Pandroidx.validateNoUnrecognizedMessages :help
```

### Exempting new build output messages

Please avoid exempting new build output and instead fix or suppress the warnings
themselves, because that will take effect not only on the build server but also
in Android Studio, and will also run more quickly.

If you cannot prevent the message from being generating and must exempt the
message anyway, follow the instructions in the error:

```shell
$ ./gradlew -Pandroidx.validateNoUnrecognizedMessages :help

Error: build_log_simplifier.py found 15 new messages found in /usr/local/google/workspace/aosp-androidx-git/out/dist/gradle.log.

Please fix or suppress these new messages in the tool that generates them.
If you cannot, then you can exempt them by doing:

  1. cp /usr/local/google/workspace/aosp-androidx-git/out/dist/gradle.log.ignore /usr/local/google/workspace/aosp-androidx-git/frameworks/support/development/build_log_simplifier/messages.ignore
  2. modify the new lines to be appropriately generalized
```

Each line in this exemptions file is a regular expressing matching one or more
lines of output to be exempted. You may want to make these expressions as
specific as possible to ensure that the addition of new, similar messages will
also be detected (for example, discovering an existing warning in a new source
file).
