/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package androidx.health.connect.client.records

import androidx.annotation.FloatRange
import androidx.health.connect.client.aggregate.AggregateMetric
import androidx.health.connect.client.metadata.Metadata
import java.time.Instant
import java.time.ZoneOffset

/**
 * Captures the user's speed, e.g. during running or cycling. Each record represents a series of
 * measurements.
 */
public class SpeedSeries(
    override val startTime: Instant,
    override val startZoneOffset: ZoneOffset?,
    override val endTime: Instant,
    override val endZoneOffset: ZoneOffset?,
    override val samples: List<Speed>,
    override val metadata: Metadata = Metadata.EMPTY,
) : SeriesRecord<Speed> {

    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is SpeedSeries) return false

        if (startTime != other.startTime) return false
        if (startZoneOffset != other.startZoneOffset) return false
        if (endTime != other.endTime) return false
        if (endZoneOffset != other.endZoneOffset) return false
        if (samples != other.samples) return false
        if (metadata != other.metadata) return false

        return true
    }

    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun hashCode(): Int {
        var result = startTime.hashCode()
        result = 31 * result + (startZoneOffset?.hashCode() ?: 0)
        result = 31 * result + endTime.hashCode()
        result = 31 * result + (endZoneOffset?.hashCode() ?: 0)
        result = 31 * result + samples.hashCode()
        result = 31 * result + metadata.hashCode()
        return result
    }

    companion object {
        private const val SPEED_TYPE_NAME = "Speed"
        private const val SPEED_FIELD_NAME = "speed"

        /**
         * Metric identifier to retrieve average speed from
         * [androidx.health.connect.client.aggregate.AggregationResult].
         */
        @JvmField
        val SPEED_AVG: AggregateMetric<Double> =
            AggregateMetric.doubleMetric(
                SPEED_TYPE_NAME,
                AggregateMetric.AggregationType.AVERAGE,
                SPEED_FIELD_NAME
            )

        /**
         * Metric identifier to retrieve minimum speed from
         * [androidx.health.connect.client.aggregate.AggregationResult].
         */
        @JvmField
        val SPEED_MIN: AggregateMetric<Double> =
            AggregateMetric.doubleMetric(
                SPEED_TYPE_NAME,
                AggregateMetric.AggregationType.MINIMUM,
                SPEED_FIELD_NAME
            )

        /**
         * Metric identifier to retrieve maximum speed from
         * [androidx.health.connect.client.aggregate.AggregationResult].
         */
        @JvmField
        val SPEED_MAX: AggregateMetric<Double> =
            AggregateMetric.doubleMetric(
                SPEED_TYPE_NAME,
                AggregateMetric.AggregationType.MAXIMUM,
                SPEED_FIELD_NAME
            )
    }
}

/**
 * Represents a single measurement of the speed, a scalar magnitude.
 *
 * @param time The point in time when the measurement was taken.
 * @param metersPerSecond Speed in meters per second. Valid range: 0-1000000.
 *
 * @see SpeedSeries
 */
public class Speed(
    val time: Instant,
    @FloatRange(from = 0.0, to = 1_000_000.0) val metersPerSecond: Double,
) {

    init {
        requireNonNegative(value = metersPerSecond, name = "metersPerSecond")
    }

    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is Speed) return false

        if (time != other.time) return false
        if (metersPerSecond != other.metersPerSecond) return false

        return true
    }

    /*
     * Generated by the IDE: Code -> Generate -> "equals() and hashCode()".
     */
    override fun hashCode(): Int {
        var result = time.hashCode()
        result = 31 * result + metersPerSecond.hashCode()
        return result
    }
}
