/*
 * Copyright 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.foundation.text

import android.os.Build
import android.view.View
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.requiredSize
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.SolidColor
import androidx.compose.ui.input.pointer.PointerIcon
import androidx.compose.ui.platform.LocalView
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.test.ExperimentalTestApi
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performMouseInput
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.unit.dp
import androidx.test.ext.junit.runners.AndroidJUnit4
import androidx.test.filters.MediumTest
import androidx.test.filters.SdkSuppress
import com.google.common.truth.Truth.assertThat
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith

@MediumTest
@OptIn(ExperimentalTestApi::class)
@SdkSuppress(minSdkVersion = Build.VERSION_CODES.N)
@RunWith(AndroidJUnit4::class)
class CoreTextFieldHoverTest {
    @get:Rule
    val rule = createComposeRule()

    private val defaultIcon = android.view.PointerIcon.TYPE_DEFAULT
    private val defaultSelectableIcon = android.view.PointerIcon.TYPE_TEXT

    /**
     * Verifies the default [PointerIcon] for [CoreTextField].
     */
    @Test
    fun IBeamDefaults() {
        val parentIconTag = "myParentIcon"
        val coreTextFieldTag = "myCoreTextField"
        val value = TextFieldValue("initial text")
        lateinit var view: View

        rule.setContent {
            view = LocalView.current
            Box(
                modifier = Modifier
                    .requiredSize(200.dp)
                    .border(BorderStroke(2.dp, SolidColor(Color.Red)))
                    .testTag(parentIconTag)
            ) {
                CoreTextField(
                    value = value,
                    onValueChange = {},
                    modifier = Modifier
                        .requiredSize(50.dp)
                        .testTag(coreTextFieldTag)
                )
            }
        }
        // Hover over CoreTextField
        rule.onNodeWithTag(coreTextFieldTag).performMouseInput {
            enter(bottomRight)
        }
        // Verify CoreTextField has default text icon
        verifyIcon(defaultSelectableIcon, view)
        // Move cursor to hover over portion of the parent box not covered by any descendants
        rule.onNodeWithTag(parentIconTag).performMouseInput {
            moveTo(bottomRight)
        }
        // Verify the current icon is the default arrow icon
        verifyIcon(defaultIcon, view)
        // Exit hovering over element
        rule.onNodeWithTag(parentIconTag).performMouseInput {
            exit()
        }
    }

    private fun verifyIcon(type: Int, view: View) {
        rule.runOnIdle {
            assertThat(view.pointerIcon).isEqualTo(
                android.view.PointerIcon.getSystemIcon(
                    view.context,
                    type
                )
            )
        }
    }
}
