# /=====================================================================\ #
# |  LaTeXML::Post::MathPath                                            | #
# | Tool for finding math tokens in formula                             | #
# |=====================================================================| #
# | Part of LaTeXML:                                                    | #
# |  Public domain software, produced as part of work done by the       | #
# |  United States Government & not subject to copyright in the US.     | #
# |---------------------------------------------------------------------| #
# | Bruce Miller <bruce.miller@nist.gov>                        #_#     | #
# | http://dlmf.nist.gov/LaTeXML/                              (o o)    | #
# \=========================================================ooo==U==ooo=/ #

# ================================================================================
# LaTeXML::Post::MathPath  
#  Translate (eventually?)(a subset of?) math expressions, in the form of
# the text attribute generated by MathParser, into XPath expressions
# which will select the appropriate XMath nodes.
# ================================================================================
package LaTeXML::Post::MathPath;
use strict;
#use LaTeXML::Post::MathDictionary;
use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = (qw(&constructMathPath));
our @EXPORT = (qw(&constructMathPath));

our %table;
our $nsURI = "http://dlmf.nist.gov/LaTeXML";

# Construct an XPath selector from a textual math expression.
# Returns undef if it can't understand the expression.
# Options are:
#   undeclared=>1 only selects nodes that don't have POS (part of speech) attribute.
#   container=>1 selects XMath nodes that have matching nodes, 
#     rather than the nodes themselves.
#   label=>$label only selects nodes within some element labeled by $label
#   refnum=>$refnum only selects nodes within some element with refnum = $refnum
sub constructMathPath {
  my($pattern,%options)=@_;
  my $nested = $pattern;
  # Replace ( and ) by markers noting the nesting level.
  my $level=0;
  $nested =~ s/(\(|\))/{ ($1 eq "\(" ? "#O".++$level."#" : "#C".$level--."#"); }/ge;
  my @xpaths = constructXPath1($nested);
  return undef unless scalar(@xpaths)==1; # Too many parts or not enough??
  my $xpath = $xpaths[0];
  return undef unless defined $xpath;

  # Add outer XPath to restrict to certain kinds of expressions

  # Restrict to Undeclared items. (NOT in XMWrap and NO POS attribute)
  $xpath .= "[not(ancestor::*".isElement('XMWrap').")][not(\@POS)]" if $options{undeclared};
  # Restrict to certain fonts
  $xpath .= "[\@font='$options{font}']" if defined $options{font};
  # Wrap to select the whole containing XMath, or just the matching node.
  $xpath = "//*". ($options{container}
		   ? isElement('XMath')."[descendant::".$xpath."]"
		   : $xpath);
  # Add restrictions for  label or refnum.
  $xpath .= "[ancestor-or-self::*[\@label='$options{label}']]" if defined $options{label};
  $xpath .= "[ancestor-or-self::*[\@refnum='$options{refnum}']]" if defined $options{refnum};

#print STDERR "$pattern => $xpath\n";
  $xpath; }

# If we can work something out for infix, maybe we can leverage the
# stuff in MathDictionary!

BEGIN{
  %table=(subscript=>\&dosubscript,
	      map(($_=>\&doaccent), (qw(OverHat OverCheck OverBreve OverAcute OverGrave 
					OverTilde OverBar OverArrow
					OverDot OverDoubleDot OverLine OverBrace
					UnderLine UnderBrace))));

}

#======================================================================
# Low-level XPath constructors.
# These all return a Test (ie in [])

# The test to check for a LaTeXML element.
sub isElement {
  my($tag)=@_;
  "[local-name()='$tag' and namespace-uri()='$nsURI']"; }

sub hasName {
  my($name)=@_;
  "[\@name='$name' or text()='$name']"; }

sub atPos {
  my($n)=@_;
  "[position()=$n]"; }

sub hasChild {
  my($test)=@_;
  "[child::*".$test."]"; }

sub hasArg {
  my($n,$test)=@_;
  hasChild(atPos($n).isElement('XMArg').hasChild($test)); }

#======================================================================

sub constructXPath1 {
  my($expr)=@_;
  my @stuff;
  do {
    #  word(...)  : some special construct
    if($expr =~ s/^(\w+)#O(\d+)#(.*)#C\2#//){
      my($op,$args)=($1,$3);
      my $fcn = $table{$op};
      my @args = constructXPath1($args);
      return undef if !defined $op or grep(!defined $_,@args);
      push(@stuff,&$fcn($op,@args)); }
    # "$"  => wildcard
    elsif($expr =~ s/^\$//){
     push(@stuff,"[true()]"); }
    # word or single char  : a specific token with the given name
    elsif($expr =~ s/^(\w+|.)//){
      my $name = $1;
     push(@stuff, isElement('XMTok').hasName($name)); }
    else { 
      return undef;	}	# Unmatched stuff? 
    } while($expr =~ s/^,//);

  @stuff; }

# This matches $base IFF it is followed by a postfix subscript operator.
# Further, if $script is defined, the subscript much match
sub dosubscript {
  my($op,$base,$script)=@_;
  $base
    ."[following-sibling::*".isElement('XMApp').hasName('PostSubscript')
      .(defined $script ? hasArg(1,$script) : '') ."]"; }

sub doaccent {
  my($op,$var)=@_;
  isElement('XMApp') 
    . hasChild(atPos(1).isElement('XMTok').hasName($op))
      . hasArg(2,$var); }

# ================================================================================
1;
