# /=====================================================================\ #
# |  LaTeXML::Post::MathML                                              | #
# | MathML generator for LaTeXML                                        | #
# |=====================================================================| #
# | Part of LaTeXML:                                                    | #
# |  Public domain software, produced as part of work done by the       | #
# |  United States Government & not subject to copyright in the US.     | #
# |---------------------------------------------------------------------| #
# | Bruce Miller <bruce.miller@nist.gov>                        #_#     | #
# | http://dlmf.nist.gov/LaTeXML/                              (o o)    | #
# \=========================================================ooo==U==ooo=/ #

# ================================================================================
# LaTeXML::MathML  Math Formatter for LaTeXML's Parsed Math.
#   Cooperate with the parsed math structure generated by LaTeXML::Math and
# convert into presentation MathML.
# ================================================================================
# TODO
#  * Need switches for Presentation and/or Content 

#  * merging of mrows when operator is `close enough' (eg (+ (+ a b) c) => (+ a b c)
#  * get presentation from DUAL
#  * proper parenthesizing (should I record the parens used when parsing?)
# Some clarity to work out:
#  We're trying to convert either parsed or unparsed math (sometimes intertwined).
# How clearly do these have to be separated?
# ================================================================================

package LaTeXML::Post::MathML;
use strict;
use LaTeXML::Util::LibXML;
use base qw(LaTeXML::Post);

our $mmlURI = "http://www.w3.org/1998/Math/MathML";

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# See END for specific converters.
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Top level
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
sub process {
  my($self,$doc)=@_;
  local $LaTeXML::Post::MathML::DOCUMENT = $doc;
  if(my @maths = $self->find_math_nodes($doc)){
    $self->Progress("Converting ".scalar(@maths)." formulae");
    $doc->addNamespace($mmlURI,'m');
    foreach my $math (@maths){
      $self->processNode($doc,$math); }
    $doc->adjust_latexml_doctype('MathML'); } # Add MathML if LaTeXML dtd.
  $doc; }

sub find_math_nodes {  $_[1]->findnodes('//ltx:Math'); }

# $self->processNode($doc,$mathnode) is the top-level conversion
# It converts the XMath within $mathnode, and adds it to the $mathnode,
sub processNode {
  my($self,$doc,$math)=@_;
  my $mode = $math->getAttribute('mode')||'inline';
  my $xmath = $doc->findnode('ltx:XMath',$math);
  my $style = ($mode eq 'display' ? 'display' : 'text');
  $doc->addNodes($math,$self->translateNode($doc,$xmath,$style,'ltx:Math')); }

# $self->translateNode($doc,$XMath,$style,$embedding)
# returns the translation of the XMath node (but doesn't insert it)
# $style will be either 'display' or 'text' (if relevant),
# The result should be wrapped as necessary for the result to
# be embedded within the tag $embedding.
# Eg. for parallel markup.

# See END for presentation, content and parallel versions.

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# General translation utilities.
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sub getTokenMeaning {
  my($node)=@_;
  my $m = $node->getAttribute('meaning') || $node->getAttribute('name')
    || $node->textContent;
  (defined $m ? $m : '?'); }

sub realize {
  my($node)=@_;
  $LaTeXML::Post::MathML::DOCUMENT->realizeXMNode($node); }

# For a node that is a (possibly embellished) operator,
# find the underlying role.
our %EMBELLISHING_ROLE=(SUPERSCRIPTOP=>1,SUBSCRIPTOP=>1,STACKED=>1,
			OVERACCENT=>1,UNDERACCENT=>1,MODIFIER=>1,MODIFIEROP=>1);
sub getOperatorRole {
  my($node)=@_;
  if(!$node){
    undef; }
  elsif(my $role = $node->getAttribute('role')){
    $role; }
  elsif($node->localname eq 'XMApp'){
    my($op,$base)= element_nodes($node);
    ($EMBELLISHING_ROLE{$op->getAttribute('role')||''}
     ? getOperatorRole($base)
     : undef); }}

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Table of Translators for presentation|content
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# All translators take XMath XML::LibXML nodes as arguments,
# and return an intermediate form of MathML to be added.

our $MMLTable_P={};
our $MMLTable_C={};

sub DefMathML {
  my($key,$presentation,$content) =@_;
  $$MMLTable_P{$key} = $presentation if $presentation;
  $$MMLTable_C{$key} = $content if $content; }

sub lookupPresenter {
  my($mode,$role,$name)=@_;
  $name = '?' unless $name;
  $role = '?' unless $role;
  $$MMLTable_P{"$mode:$role:$name"} || $$MMLTable_P{"$mode:?:$name"}
    || $$MMLTable_P{"$mode:$role:?"} || $$MMLTable_P{"$mode:?:?"}; }

sub lookupContent {
  my($mode,$role,$name)=@_;
  $name = '?' unless $name;
  $role = '?' unless $role;
  $$MMLTable_C{"$mode:$role:$name"} || $$MMLTable_C{"$mode:?:$name"}
    || $$MMLTable_C{"$mode:$role:?"} || $$MMLTable_C{"$mode:?:?"}; }


#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Support functions for Presentation MathML
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sub pmml_top {
  my($node,$style)=@_;
  local $LaTeXML::MathML::STYLE = $style;
  local $LaTeXML::MathML::FONT  = find_inherited_attribute($node,'font');
  local $LaTeXML::MathML::SIZE  = find_inherited_attribute($node,'size');
  local $LaTeXML::MathML::COLOR = find_inherited_attribute($node,'color');
  pmml($node); }

sub find_inherited_attribute {
  my($node,$attribute)=@_;
  while($node && isElementNode($node)){
    if(my $value = $node->getAttribute($attribute)){
      return $value; }
    $node = $node->parentNode; }
  return undef; }

our %stylestep=(display=>'text', text=>'script',
	       script=>'scriptscript', scriptscript=>'scriptscript');
our %stylemap
  = (display     =>{text        =>[displaystyle=>'false'],
		    script      =>[displaystyle=>'false',scriptlevel=>'+1'],
		    scriptscript=>[displaystyle=>'false',scriptlevel=>'+2']},
     text        =>{display     =>[displaystyle=>'true'],
		    script      =>[scriptlevel=>'+1'],
		    scriptscript=>[scriptlevel=>'+2']},
     script      =>{display     =>[displaystyle=>'true',scriptlevel=>'-1'],
		    text        =>[scriptlevel=>'-1'],
		    scriptscript=>[scriptlevel=>'+1']},
     scriptscript=>{display     =>[displaystyle=>'true',scriptlevel=>'-2'],
		    text        =>[scriptlevel=>'-2'],
		    script      =>[scriptlevel=>'-1']});

sub pmml_smaller {
  my($node)=@_;
  local $LaTeXML::MathML::STYLE = $stylestep{$LaTeXML::MathML::STYLE};
  pmml($node); }

sub pmml {
  my($node)=@_;
  my $o = $node->getAttribute('open');
  my $c = $node->getAttribute('close');
  my $p = $node->getAttribute('punctuation');
  # Do the core conversion.
  my $result = ($node->localname eq 'XMRef'
		? pmml(realize($node))
		: pmml_internal($node));
  # Handle generic things: open/close delimiters, punctuation
  $result = pmml_parenthesize($result,$o,$c) if $o || $c;
  $result = ['m:mrow',{},$result,pmml_mo($p)] if $p;
  $result; }

our $NBSP = pack('U',0xA0);

sub pmml_internal {
  my($node)=@_;
  return ['m:merror',{},['m:mtext',{},"Missing Subexpression"]] unless $node;
  my $tag = $node->localname;
  my $role = $node->getAttribute('role');
  if($tag eq 'XMath'){
    pmml_row(map(pmml($_), element_nodes($node))); } # Really multiple nodes???
  elsif($tag eq 'XMDual'){
    my($content,$presentation) = element_nodes($node);
    pmml($presentation); }
  elsif($tag eq 'XMWrap'){	# Only present if parsing failed!
    pmml_row(map(pmml($_),element_nodes($node))); }
  elsif($tag eq 'XMApp'){
    my($op,@args) = element_nodes($node);
    if(!$op){
      ['m:merror',{},['m:mtext',{},"Missing Operator"]]; }
    elsif($role && ($role =~ /^(FLOAT|POST)(SUB|SUPER)SCRIPT$/)){
      pmml_unparsed_script($1,$2,$op); }
    else {
      my $rop = realize($op);  # NOTE: Could loose open/close on XMRef ???
      my $style = $op->getAttribute('style');
      my $styleattr = $style && $stylemap{$LaTeXML::MathML::STYLE}{$style};
      local $LaTeXML::MathML::STYLE 
	= ($style && $stylestep{$style} ? $style : $LaTeXML::MathML::STYLE);
      my $result = &{ lookupPresenter('Apply',getOperatorRole($rop),getTokenMeaning($rop))
		    }($op,@args);
      $result = ['m:mstyle',{@$styleattr},$result] if $styleattr;
      $result; }}
  elsif($tag eq 'XMTok'){
    &{ lookupPresenter('Token',$role,getTokenMeaning($node)) }($node); }
  elsif($tag eq 'XMHint'){
    &{ lookupPresenter('Hint',$role,getTokenMeaning($node)) }($node); }
  elsif($tag eq 'XMArray'){
    my $style = $node->getAttribute('style');
    my $styleattr = $style && $stylemap{$LaTeXML::MathML::STYLE}{$style};
    local $LaTeXML::MathML::STYLE 
      = ($style && $stylestep{$style} ? $style : $LaTeXML::MathML::STYLE);
    my @rows = ();
    foreach my $row (element_nodes($node)){
      my @cols = ();
      foreach my $col (element_nodes($row)){
	my $a = $col->getAttribute('align');
	my $b = $col->getAttribute('border');
	my $h = (($col->getAttribute('thead')||'') eq 'yes') && 'thead';
	my $c = ($b ? ($h ? "$b $h" : $b) : $h);
	my $cs = $col->getAttribute('colspan');
	my $rs = $col->getAttribute('rowspan');
	push(@cols,['m:mtd',{($a ? (columnalign=>$a):()),
			     ($c ? (class=>$c):()),
			     ($cs ? (columnspan=>$cs):()),
			     ($rs ? (rowspan=>$rs):())},
		    map(pmml($_),element_nodes($col))]); }
      push(@rows,['m:mtr',{},@cols]); }
    my $result = ['m:mtable',{rowspacing=>"0.2ex", columnspacing=>"0.4em"},@rows];
    $result = ['m:mstyle',{@$styleattr},$result] if $styleattr;
    $result; }
  else {
    my $text = $node->textContent; #  Spaces are significant here
    $text =~ s/^\s+/$NBSP/;
    $text =~ s/\s+$/$NBSP/;
    ['m:mtext',{},$text]; }}

sub pmml_row {
  my(@items)=@_;
  @items = grep($_,@items);
  (scalar(@items) == 1 ? $items[0] : ['m:mrow',{},@items]); }

sub pmml_unrow {
  my($mml)=@_;
  if($mml && (ref $mml)  && ($mml->[0] eq 'm:mrow') && !scalar(keys %{$mml->[1]})){
    my($tag,$attr,@children)=@$mml;
    @children; }
  else {
    ($mml); }}

sub pmml_parenthesize {
  my($item,$open,$close)=@_;
  if(!$open && !$close){
    $item; }
## Maybe better not open the contained mrow; seems to affect bracket size in Moz.
##  elsif($item && (ref $item)  && ($item->[0] eq 'mrow')){
##    my($tag,$attr,@children)=@$item;
##    ['m:mrow',$attr,($open ? (pmml_mo($open)):()),@children,($close ? (pmml_mo($close)):())]; }
  else {
    ['m:mrow',{},($open ? (pmml_mo($open)):()),$item,($close ? (pmml_mo($close)):())]; }}

sub pmml_punctuate {
  my($separators,@items)=@_;
  $separators='' unless defined $separators;
  my $lastsep=', ';
  my @arglist;
  if(@items){
    push(@arglist,pmml(shift(@items)));
    while(@items){
      $separators =~ s/^(.)//;
      $lastsep = $1 if $1;
      push(@arglist,pmml_mo($lastsep),pmml(shift(@items))); }}
  pmml_row(@arglist); }


# args are XMath nodes
sub pmml_infix {
  my($op,@args)=@_;
  $op = realize($op);
  return ['m:mrow',{}] unless $op && @args; # ??
  my @items=();
  if(scalar(@args) == 1){	# Infix with 1 arg is presumably Prefix!
    push(@items,(ref $op ? pmml($op) : pmml_mo($op)),pmml($args[0])); }
  else {
    push(@items, pmml(shift(@args)));
    while(@args){
      push(@items,(ref $op ? pmml($op) : pmml_mo($op)));
      push(@items,pmml(shift(@args))); }}
  pmml_row(@items); }

# Mappings between internal fonts & sizes.
# Default math font is roman|medium|upright.
our %mathvariants = ('upright'          =>'normal',
		     'bold'             =>'bold',
		     'bold upright'     =>'bold',
		     'italic'           =>'italic',
		     'medium italic'    =>'italic',
		     'bold italic'      =>'bold-italic',
		     'doublestruck'     =>'double-struck',
		     'blackboard'       =>'double-struck',
		     'fraktur bold'     => 'bold-fraktur',
		     'script'           => 'script',
		     'script italic'    => 'script',
		     'script bold'      => 'bold-script',
		     'caligraphic'      => 'script',
		     'caligraphic bold' => 'bold-script',
		     'fraktur'          => 'fraktur',
		     'sansserif'        => 'sans-serif',
		     'sansserif bold'   => 'bold-sans-serif',
		     'sansserif italic' => 'sans-serif-italic',
		     'sansserif bold italic'   => 'sans-serif-bold-italic',
		     'typewriter'       => 'monospace');

# The font differences (from the containing context) have been deciphered
# into font, size and color attributes.  The font should match
# one of the above... (?)

our %sizes=(tiny=>'small',script=>'small',footnote=>'small',small=>'small',
	    normal=>'normal',
	    large=>'big',Large=>'big',LARGE=>'big',huge=>'big',Huge=>'big');

sub pmml_mi {
  my($item)=@_;
  my $font  = (ref $item ? $item->getAttribute('font') : undef) ||  $LaTeXML::MathML::FONT;
  my $size  = (ref $item ? $item->getAttribute('size') : undef) || $LaTeXML::MathML::SIZE;
  my $color = (ref $item ? $item->getAttribute('color') : undef) || $LaTeXML::MathML::COLOR;
  my $text  = (ref $item ?  $item->textContent : $item);
  my $variant = ($font ? $mathvariants{$font} : '');
  if($font && !$variant){
    warn "Unrecognized font variant \"$font\""; $variant=''; }
  if($text =~ /^.$/){	# Single char in mi?
    if($variant eq 'italic'){ $variant = ''; } # Defaults to italic
    elsif(!$variant){ $variant = 'normal'; }}  # must say so explicitly.
  ['m:mi',{($variant ? (mathvariant=>$variant):()),
	   ($size    ? (mathsize=>$sizes{$size}):()),
	   ($color   ? (mathcolor=>$color):())},$text]; }

sub pmml_mo {
  my($item)=@_;
  my $font  = (ref $item ? $item->getAttribute('font') : undef);
  my $size  = (ref $item ? $item->getAttribute('size') : undef);
  my $color = (ref $item ? $item->getAttribute('color') : undef);
  my $text  = (ref $item ?  $item->textContent : $item);
  my $variant = ($font ? $mathvariants{$font} : '');
  my $pos   = (ref $item && $item->getAttribute('scriptpos')) || 'post';
  ['m:mo',{($variant ? (mathvariant=>$variant):()),
	   ($size    ? (mathsize=>$sizes{$size}):()),
	   ($color   ? (mathcolor=>$color):()),
	   # If an operator has specifically located it's scripts,
	   # don't let mathml move them.
	   (($pos =~ /mid/) || $LaTeXML::MathML::NOMOVABLELIMITS
	    ? (movablelimits=>'false'):())},
   $text]; }

## (FLOAT|POST)(SUB|SUPER)SCRIPT's should NOT remain in successfully parsed math.
# This gives something `presentable', though not correct.
# What to use for base? I can't reasonably go up & grap the preceding token...
# I doubt an empty <mi/> is valid, but what is?
sub pmml_unparsed_script {
  my($x,$y,$script)=@_;
  [ ($y eq 'SUB' ? 'm:msub' : 'm:msup' ), {}, ['m:mi'],
    pmml_smaller($script)]; }

sub pmml_script {
  my($script)=@_;
  ($script ? pmml_smaller($script) : ['m:none']); }

# Since we're keeping track of display style, under/over vs. sub/super
# We've got to override MathML's desire to do it for us.
# Here, we make sure the eventual inner operator (if any) has
# movablelimits disabled.
# NOTE: Another issue is when the base is "embellished", in particular
# has sub/superscripts of it's own.
# Mozilla (at least?) centers the over/under wrong in that case.
# The OVERUNDERHACK makes the sub & superscripts have 0 width 
# in this situation.
# Worried that this will end up biting me, though...
sub do_overunder {
  my($tag,$base,@scripts)=@_;
  { local $LaTeXML::MathML::NOMOVABLELIMITS=1;
    local $LaTeXML::MathML::OVERUNDERHACKS=1;
    $base = pmml($base); }
  my $form = [$tag,{},$base,map(pmml_smaller($_),@scripts)];
#  if($LaTeXML::MathML::STYLE ne 'display'){ # Workaround Mozilla bug (?)
#    ['m:mstyle',{displaystyle=>'false'},$form]; }
#  else {
    $form; }
#}

sub do_subsup {
  my($tag,$base,@scripts)=@_;
  $base = pmml($base);
  @scripts = map(pmml_smaller($_),@scripts);
  if($LaTeXML::MathML::OVERUNDERHACKS){
    @scripts = map(['m:mpadded',{width=>'0'},$_],@scripts); }
  [$tag,{},$base,@scripts]; }

sub pmml_script_handler {
  my($op,$base,$script)=@_;
  my(@pres,@posts);
  my($prelevel,$postlevel)=(0,0);
  my ($y) = ($op->getAttribute('role')||'') =~ /^(SUPER|SUB)SCRIPTOP$/;
  my ($x,$l)= ($op->getAttribute('scriptpos')||'post0')
    =~ /^(pre|mid|post)?(\d+)?$/;
  if($x eq 'pre'){
    if($y eq 'SUB'){
      push(@pres,[$script,undef]); $prelevel=$l; }
    elsif($y eq 'SUPER'){
      push(@pres,[undef,$script]); $prelevel=$l; }}
  else {
    if($y eq 'SUB'){
      push(@posts,[$script,undef]); $postlevel=$l; }
    elsif($y eq 'SUPER'){
      push(@posts,[undef,$script]); $postlevel=$l; }}

  # Keep from having multiple scripts when $loc is stack!!!
  while(1){
    last unless $base->localname eq 'XMApp';
    my($xop,$xbase,$xscript) = element_nodes($base);
    last unless ($xop->localname eq 'XMTok');
    my ($ny) = ($xop->getAttribute('role')||'') =~ /^(SUPER|SUB)SCRIPTOP$/;
    last unless $ny;
    my ($nx,$nl)= ($xop->getAttribute('scriptpos')||'postsup0')
      =~ /^(pre|mid|post)?(\d+)?$/;
    last unless ($x ne 'mid') || ($nx eq 'mid');

    my $spos = ($ny eq 'SUB' ? 0 : 1);
    if($nx eq 'pre'){
      push(@pres,[undef,undef]) # New empty pair (?)
	if($prelevel ne $nl) || $pres[-1][$spos];
      $pres[-1][$spos] = $xscript; $prelevel = $nl; }
    else {
      unshift(@posts,[undef,undef]) # New empty pair (?)
	if($postlevel ne $nl) || $posts[0][$spos];
      $posts[0][$spos] = $xscript; $postlevel = $nl; }
    $base = $xbase;
  }
  if(scalar(@pres) > 0){
    ['m:mmultiscripts',{},
     pmml($base),
     map( (pmml_script($_->[0]),pmml_script($_->[1])), @posts),
     ['m:mprescripts'],
     map( (pmml_script($_->[0]),pmml_script($_->[1])), @pres)]; }
  elsif(scalar(@posts) > 1){
    ['m:mmultiscripts',{},
     pmml($base),
     map( (pmml_script($_->[0]),pmml_script($_->[1])), @posts)]; }
  elsif(!defined $posts[0][1]){
    if($x eq 'mid'){ do_overunder('m:munder',$base,$posts[0][0]); }
    else           { do_subsup('m:msub',$base,$posts[0][0]); }}
  elsif(!defined $posts[0][0]){
    if($x eq 'mid'){ do_overunder('m:mover',$base,$posts[0][1]); }
    else           { do_subsup('m:msup',$base,$posts[0][1]); }}
  else {
    if($x eq 'mid'){ do_overunder('m:munderover',$base,$posts[0][0],$posts[0][1]); }
    else           { do_subsup('m:msubsup',$base,$posts[0][0],$posts[0][1]); }}}

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Support functions for Content MathML
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sub cmml_top {
  my($node)=@_;
  cmml($node); }

sub cmml {
  my($node)=@_;
  return ['m:merror',{},['m:mtext',{},"Missing Subexpression"]] unless $node;
  $node = realize($node) if $node->localname eq 'XMRef';
  my $tag = $node->localname;
  if($tag eq 'XMath'){
    my($item,@rest)=  element_nodes($node);
    print STDERR "Warning! got extra nodes for content!\n" if @rest;
    cmml($item); }
  elsif($tag eq 'XMDual'){
    my($content,$presentation) = element_nodes($node);
    cmml($content); }
  elsif($tag eq 'XMWrap'){	# Only present if parsing failed!
    pmml_row(map(pmml($_),element_nodes($node))); } # ????
  elsif($tag eq 'XMApp'){
    my($op,@args) = element_nodes($node);
    if(!$op){
      ['m:merror',{},['m:mtext',{},"Missing Operator"]]; }
    else {
      my $rop = realize($op);		# NOTE: Could loose open/close on XMRef ???
      &{ lookupContent('Apply',$rop->getAttribute('role'),getTokenMeaning($rop)) }($op,@args); }}
  elsif($tag eq 'XMTok'){
    &{ lookupContent('Token',$node->getAttribute('role'),getTokenMeaning($node)) }($node); }
  elsif($tag eq 'XMHint'){	# ????
    &{ lookupContent('Hint',$node->getAttribute('role'),getTokenMeaning($node)) }($node); }
  else {
    ['m:mtext',{},$node->textContent]; }}

# Or csymbol if there's some kind of "defining" attribute?
sub cmml_ci {
  my($item)=@_;
  my $font    = (ref $item ? $item->getAttribute('font') : undef);
  my $variant = ($font && $mathvariants{$font})||'';
  my $content = (ref $item ?  $item->textContent : $item);
  if($content =~ /^.$/){	# Single char?
    if($variant eq 'italic'){ $variant = ''; } # Defaults to italic
    elsif(!$variant){ $variant = 'normal'; }}  # must say so explicitly.
#  ['m:csymbol',{($variant ? (mathvariant=>$variant) : ())},$content]; }
  ['m:ci',{},$content]; }

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Tranlators
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Organized according to the MathML Content element lists.
# As a general rule, presentation conversions are based on role
# (eg "Token:role:?"), whereas content conversions are based
# on meaning or name (eg. "Token:?:meaning").

#======================================================================
# Token elements:
#   cn, ci, csymbol

DefMathML("Token:?:?",           \&pmml_mi, \&cmml_ci);
DefMathML("Token:PUNCT:?",       \&pmml_mo, undef);
DefMathML("Token:PERIOD:?",      \&pmml_mo, undef);
DefMathML("Token:OPEN:?",        \&pmml_mo, undef);
DefMathML("Token:CLOSE:?",       \&pmml_mo, undef);
DefMathML("Token:MIDDLE:?",      \&pmml_mo, undef);
DefMathML("Token:VERTBAR:?",     \&pmml_mo, undef);
DefMathML("Token:ARROW:?",       \&pmml_mo, undef);
DefMathML("Token:OVERACCENT:?",  \&pmml_mo, undef);
DefMathML("Token:UNDERACCENT:?", \&pmml_mo, undef);

DefMathML("Token:NUMBER:?",
	  sub { ['m:mn',{},$_[0]->textContent]; },
	  sub { ['m:cn',{},$_[0]->textContent]; });
DefMathML("Token:?:Empty", sub { ['m:none']} );
DefMathML('Hint:?:?', sub { undef; }, sub { undef; }); # Should Disappear!

# At presentation level, these are essentially adorned tokens.
# args are (accent,base)
DefMathML('Apply:OVERACCENT:?', sub {
  my($accent,$base)=@_;
  ['m:mover',{accent=>'true'}, pmml($base),pmml_smaller($accent)]; });

DefMathML('Apply:UNDERACCENT:?', sub {
  my($accent,$base)=@_;
  ['m:munder',{accent=>'true'}, pmml($base),pmml_smaller($accent)]; });

#======================================================================
# Basic Content elements:
#   apply, interval, inverse, sep, condition, declare, lambda, compose, ident,
#   domain, codomain, image, domainofapplication, piecewise, piece, otherwise

# BRM
DefMathML("Token:?:\x{2061}", \&pmml_mo, undef); # FUNCTION APPLICATION
DefMathML("Token:OPERATOR:?", \&pmml_mo, undef);

DefMathML('Apply:?:?', sub {
  my($op,@args)=@_;
  ['m:mrow',{},
   pmml($op),pmml_mo("\x{2061}"),	# FUNCTION APPLICATION
   pmml_parenthesize(pmml_punctuate($op->getAttribute('separators'),@args),
		     $op->getAttribute('argopen'),$op->getAttribute('argclose'))]; },
  sub {
    my($op,@args)=@_;
    ['m:apply',{},cmml($op), map(cmml($_),@args)]; });
DefMathML('Apply:COMPOSEOP:?', \&pmml_infix, undef);


# MK
# Are these valid elements??
# Wouldn't it be better to use Apply ?
# Also coordinate with my interval symbols.
# TO MK: How about interval-cc, etc; No need for CONSTRUCTOR role (?)

DefMathML("Token:?:open_interval",       undef,sub{['m:interval',{closure=>"open"}];});
DefMathML("Token:?:closed_interval",     undef,sub{['m:interval',{closure=>"closed"}];});
DefMathML("Token:?:closed_open_interval",undef,sub{['m:interval',{closure=>"closed-open"}];});
DefMathML("Token:?:open_closed_interval",undef,sub{['m:interval',{closure=>"open-closed"}];});


DefMathML("Token:?:inverse",    undef, sub{['m:inverse'];}); # OPFUNCTION
DefMathML("Token:?:lambda",     undef, sub{['m:lambda'];}); # BINDER
DefMathML("Token:?:compose",    undef, sub{['m:compose'];}); # MULOP
DefMathML("Token:ID:ident",     undef, sub{['m:ident'];});
DefMathML("Token:?:domain",     undef, sub{['m:domain'];});	   # OPFUNCTION
DefMathML("Token:?:codomain",   undef, sub{['m:codomain'];});	   # OPFUNCTION
DefMathML("Token:?:image",      undef, sub{['m:image'];});	   # OPFUNCTION
DefMathML("Token:?:piecewise",  undef, sub{['m:piecewise'];});	   # OPFUNCTION
DefMathML("Token:?:piece",      undef, sub{['m:piece'];});	   # OPFUNCTION
DefMathML("Token:?:otherwise",  undef, sub{['m:otherwise'];});	   # OPFUNCTION

#======================================================================
# Arithmetic, Algebra and Logic:
#   quotient, factorial, divide, max, min, minus, plus, power, rem, times, root
#   gcd, and, or, xor, not, implies, forall, exists, abs, conjugate, arg, real,
#   imaginary, lcm, floor, ceiling.

# BRM:

DefMathML("Token:ADDOP:?",       \&pmml_mo,    undef);
DefMathML("Token:ADDOP:plus",     undef,       sub { ['m:plus'];});
DefMathML("Token:ADDOP:minus",    undef,       sub { ['m:minus'];});
DefMathML('Apply:ADDOP:?',       \&pmml_infix, undef);

DefMathML("Token:MULOP:?",       \&pmml_mo,    undef);
DefMathML("Token:MULOP:times",    undef,       sub { ['m:times'];});
DefMathML("Token:MULOP:\x{2062}", undef,       sub { ['m:times'];});
DefMathML("Token:MULOP:div",      undef,       sub { ['m:divide'];});
DefMathML("Token:?:\x{2062}",    \&pmml_mo,    undef); # INVISIBLE TIMES
DefMathML('Apply:MULOP:?',       \&pmml_infix, undef);
DefMathML('Apply:?:div', sub {
  my($op,$num,$den)=@_;
  my $style = $op->getAttribute('style');
  my $thickness = $op->getAttribute('thickness');
#  ['m:mfrac',{($thickness ? (linethickness=>$thickness):()),
#	    ($style && ($style eq 'inline') ? (bevelled=>'true'):())},
#   pmml_smaller($num),pmml_smaller($den)]; });
  # Bevelled looks crappy (operands too small) in Mozilla, so just open-code it.
  if($style && ($style eq 'inline')){
    ['m:mrow',{},pmml($num),pmml_mo('/'),pmml($den)]; }
  else {
    ['m:mfrac',{($thickness ? (linethickness=>$thickness):())},
     pmml_smaller($num),pmml_smaller($den)]; }});

DefMathML("Token:SUPOP:?",        \&pmml_mo,   undef);
DefMathML('Apply:SUPERSCRIPTOP:?',\&pmml_script_handler, undef);
DefMathML('Apply:SUBSCRIPTOP:?',  \&pmml_script_handler, undef);

DefMathML('Apply:POSTFIX:?', sub {
  ['m:mrow',{},pmml($_[1]),pmml($_[0])]; });

DefMathML('Apply:?:sqrt', sub { ['m:msqrt',{},pmml($_[1])]; });
DefMathML('Apply:?:root', sub { ['m:mroot',{},pmml($_[2]),pmml_smaller($_[1])]; });

# MK:
DefMathML("Token:?:quotient",  undef, sub{['m:quotient'];}); # OPFUNCTION
DefMathML("Token:?:factorial", undef, sub{['m:factorial'];}); # OPFUNCTION
DefMathML("Token:?:divide",    undef, sub{['m:divide'];});    # OPFUNCTION
DefMathML("Token:?:max",       undef, sub{['m:max'];});	      # OPFUNCTION
DefMathML("Token:?:min",       undef, sub{['m:min'];});	      # OPFUNCTION
DefMathML("Token:?:minus",     undef, sub{['m:minus'];});     # ADDOP
DefMathML("Token:?:uminus",    undef, sub{['m:uminus'];}); # OPFUNCTION
DefMathML("Token:?:plus",      undef, sub{['m:plus'];});       # ADDOP
DefMathML("Token:?:power",     undef, sub{['m:power'];});      # OPFUNCTION
DefMathML("Token:?:rem",       undef, sub{['m:rem'];});	       # OPFUNCTION
DefMathML("Token:?:times",     undef, sub{['m:times'];});      # MULOP
DefMathML("Token:?:root",      undef, sub{['m:root'];});       # OPFUNCTION
DefMathML("Token:?:gcd",       undef, sub{['m:gcd'];});	       # OPFUNCTION
DefMathML("Token:?:and",       undef, sub{['m:and'];});	       # CONNECTIVE
DefMathML("Token:?:or",        undef, sub{['m:or'];});	       # CONNECTIVE
DefMathML("Token:?:xor",       undef, sub{['m:xor'];});	       # CONNECTIVE
DefMathML("Token:?:not",       undef, sub{['m:not'];});	       # CONNECTIVE
DefMathML("Token:?:implies",   undef, sub{['m:implies'];});    # CONNECTIVE
DefMathML("Token:?:And",       undef, sub{['m:And'];});    # BINDER
DefMathML("Token:?:Or",        undef, sub{['m:Or'];});     # BINDER
DefMathML("Token:?:Xor",       undef, sub{['m:Xor'];});    # BINDER
DefMathML("Token:?:forall",    undef, sub{['m:forall'];}); # BINDER
DefMathML("Token:?:exists",    undef, sub{['m:exists'];}); # BINDER
DefMathML("Token:?:abs",       undef, sub{['m:abs'];});	       # OPFUNCTION
DefMathML("Token:?:conjugate", undef, sub{['m:conjugate'];});  # OPFUNCTION
DefMathML("Token:?:arg",       undef, sub{['m:arg'];});	       # OPFUNCTION
DefMathML("Token:?:real",      undef, sub{['m:real'];});       # OPFUNCTION
DefMathML("Token:?:imaginary", undef, sub{['m:imaginary'];});  # OPFUNCTION
DefMathML("Token:?:lcm",       undef, sub{['m:lcm'];});	       # OPFUNCTION

# Interesting question:
#  Are these "overspecified" ?
#  Maybe MK wants to write floor(x), whereas I want to write \lfloor x \rfloor
# The role should probably be ? here, so that _both_ cases translate to m:floor ???
DefMathML("Token:?:floor",     undef, sub{['m:floor'];}); # OPFUNCTION
DefMathML("Token:?:ceiling",   undef, sub{['m:ceiling'];}); # OPFUNCTION

#======================================================================
# Relations:
#   eq, neq, gt, lt, geq, leq, equivalent, approx, factorof

#BRM
DefMathML("Token:RELOP:?",         \&pmml_mo);
DefMathML("Token:RELOP:eq",         undef,     sub { ['m:eq'];});
DefMathML("Token:RELOP:\x{2260}",   undef,     sub { ['m:neq'];}); # \ne, not-eq .. ???
DefMathML("Token:RELOP:greater",    undef,     sub { ['m:gt'];});
DefMathML("Token:RELOP:less",       undef,     sub { ['m:lt'];});
DefMathML("Token:RELOP:less-eq",    undef,     sub { ['m:leq'];});
DefMathML("Token:RELOP:greater-eq", undef,   sub { ['m:geq'];});
DefMathML("Token:METARELOP:?",     \&pmml_mo);
DefMathML('Apply:RELOP:?',         \&pmml_infix);
DefMathML('Apply:METARELOP:?',     \&pmml_infix);

# Top level relations
DefMathML('Apply:?:Formulae',sub { 
  my($op,@elements)=@_;
  pmml_punctuate($op->getAttribute('separators'),@elements); });
DefMathML('Apply:?:MultiRelation',sub { 
  my($op,@elements)=@_;
  pmml_row(map(pmml($_),@elements)); });

#MK
DefMathML("Token:?:eq",        undef, sub{['m:eq'];}); # RELOP
# Hmm, neq, gt, lt ?? ???
DefMathML("Token:?:neq",       undef, sub{['m:neq'];}); # RELOP
DefMathML("Token:?:gt",        undef, sub{['m:gt'];});	# RELOP
DefMathML("Token:?:lt",        undef, sub{['m:lt'];});	# RELOP
DefMathML("Token:?:geq",       undef, sub{['m:geq'];}); # RELOP
DefMathML("Token:?:leq",       undef, sub{['m:leq'];}); # RELOP
DefMathML("Token:?:equivalent",undef, sub{['m:equivalent'];}); # RELOP
DefMathML("Token:?:approx",    undef, sub{['m:approx'];});     # RELOP
DefMathML("Token:?:factorof",  undef, sub{['m:factorof'];});   # RELOP

#======================================================================
# Calculus and Vector Calculus:
#   int, diff, partialdiff, lowlimit, uplimit, bvar, degree, 
#   divergence, grad, curl, laplacian.

# BRM
DefMathML("Token:INTOP:?",       \&pmml_mo);
DefMathML("Token:LIMITOP:?",     \&pmml_mo);
DefMathML('Apply:ARROW:?',       \&pmml_infix);

# MK
DefMathML("Token:?:int",       undef, sub{['m:int'];}); # BINDER
DefMathML("Token:?:diff",      undef, sub{['m:diff'];});    # OPFUNCTION
DefMathML("Token:?:degree",    undef, sub{['m:degree'];});  # OPFUNCTION
DefMathML("Token:?:limit",     undef, sub{['m:limit'];});   # OPFUNCTION
DefMathML("Token:?:tendsto",   undef, sub{['m:tendsto'];}); # RELOP
DefMathML("Token:?:divergence",undef, sub{['m:divergence'];});	 # OPFUNCTION
DefMathML("Token:?:grad",      undef, sub{['m:grad'];});	 # OPFUNCTION
DefMathML("Token:?:url",       undef, sub{['m:url'];});		 # OPFUNCTION
DefMathML("Token:?:laplacian", undef, sub{['m:laplacian'];});	 # OPFUNCTION

#======================================================================
# Theory of Sets,
#   set, list, union, intersect, in, notin, subset, prsubset, notsubset, notprsubset,
#   setdiff, card, cartesianproduct.

# MK
DefMathML("Token:?:set",             undef, sub{['m:set'];}); # CONSTRUCTOR
DefMathML("Token:?:list",            undef, sub{['m:list'];}); # CONSTRUCTOR
DefMathML("Token:?:union",           undef, sub{['m:union'];}); # OPFUNCTION
DefMathML("Token:?:intersect",       undef, sub{['m:intersect'];}); # OPFUNCTION
DefMathML("Token:?:in",              undef, sub{['m:in'];});   # RELOP
DefMathML("Token:?:notin",           undef, sub{['m:notin'];}); # RELOP
DefMathML("Token:?:subset",          undef, sub{['m:subset'];}); # RELOP
DefMathML("Token:?:prsubset",        undef, sub{['m:prsubset'];}); # RELOP
DefMathML("Token:?:notsubset",       undef, sub{['m:notsubset'];}); # RELOP
DefMathML("Token:?:notprsubset",     undef, sub{['m:notprsubset'];}); # RELOP
DefMathML("Token:?:setdiff",         undef, sub{['m:setdiff'];}); # OPFUNCTION
DefMathML("Token:?:card",            undef, sub{['m:card'];});    # OPFUNCTION
DefMathML("Token:?:cartesianproduct",undef, sub{['m:cartesianproduct'];}); # OPFUNCTION
DefMathML("Token:?:Union",           undef, sub{['m:Union'];}); # BINDER
DefMathML("Token:?:Union",           undef, sub{['m:Union'];}); # BINDER
DefMathML("Token:?:Intersect",       undef, sub{['m:Intersect'];}); # BINDER
DefMathML("Token:?:Cartesianproduct",undef, sub{['m:Cartesianproduct'];}); # BINDER

#======================================================================
# Sequences and Series:
#   sum, product, limit, tendsto

# (but see calculus for limit too!!)

# BRM
DefMathML("Token:SUMOP:?",       \&pmml_mo);
sub pmml_bigop {
  my($op,$body)=@_;
  ['m:mrow',{}, pmml($op), pmml_unrow(pmml($body))]; }
DefMathML('Apply:BIGOP:?',\&pmml_bigop);
DefMathML('Apply:INTOP:?',\&pmml_bigop);
DefMathML('Apply:SUMOP:?',\&pmml_bigop);

DefMathML('Apply:?:LimitFrom', sub {
  my($op,$arg,$dir)=@_;
  ['m:mrow',{},pmml($arg),pmml($dir)]; });

DefMathML('Apply:?:Annotated', sub {
  my($op,$var,$annotation)=@_;
  ['m:mrow',{},pmml($var),pmml($annotation)];});

# NOTE: Markup probably isn't right here....
DefMathML('Apply:?:AT', sub {
  my($op,$expr,$value)=@_;
  pmml_row(pmml($expr),['m:msub',{},pmml_mo('|'),pmml_smaller($value)]); });

# MK
DefMathML("Token:BINDER:sum",          undef, sub{['m:sum'];});
DefMathML("Token:BINDER:prod",         undef, sub{['m:prod'];});

#======================================================================
# Elementary Classical Functions,
#   exp, ln, log, sin, cos tan, sec, csc, cot, sinh, cosh, tanh, sech, csch, coth,
#   arcsin, arccos, arctan, arccosh, arccot, arccoth, arccsc, arccsch, arcsec, arcsech,
#   arcsinh, arctanh

# Hmm, for content, we probably should leave off the Role;
# This would allow them to be generated with different
# presentation styles, w/o affecting the content conversion.
# Providing, of course, that the meaning is consistently used!

DefMathML("Token:?:exp",      undef, sub { ['m:exp']; }); # OPFUNCTION
DefMathML("Token:?:ln",       undef, sub { ['m:ln']; });  # OPFUNCTION
DefMathML("Token:?:log",      undef, sub { ['m:log']; }); # OPFUNCTION
DefMathML("Token:?:sin",      undef, sub { ['m:sin']; }); # TRIGFUNCTION
DefMathML("Token:?:cos",      undef, sub { ['m:cos']; }); # TRIGFUNCTION
DefMathML("Token:?:tan",      undef, sub { ['m:tan']; }); # TRIGFUNCTION
DefMathML("Token:?:sec",      undef, sub { ['m:sec']; }); # TRIGFUNCTION
DefMathML("Token:?:csc",      undef, sub { ['m:csc']; }); # TRIGFUNCTION
DefMathML("Token:?:cot",      undef, sub { ['m:cot']; }); # TRIGFUNCTION
DefMathML("Token:?:sinh",     undef, sub { ['m:sinh']; });  # TRIGFUNCTION
DefMathML("Token:?:cosh",     undef, sub { ['m:cosh']; });  # TRIGFUNCTION
DefMathML("Token:?:tanh",     undef, sub { ['m:tanh']; });  # TRIGFUNCTION
DefMathML("Token:?:sech",     undef, sub { ['m:sech']; });  # TRIGFUNCTION
DefMathML("Token:?:csch",     undef, sub { ['m:csch']; });  # TRIGFUNCTION
DefMathML("Token:?:coth",     undef, sub { ['m:coth']; });  # TRIGFUNCTION
DefMathML("Token:?:arcsin",   undef, sub { ['m:arcsin']; }); # OPFUNCTION
DefMathML("Token:?:arccos",   undef, sub { ['m:arccos']; }); # OPFUNCTION
DefMathML("Token:?:arctan",   undef, sub { ['m:arctan']; }); # OPFUNCTION
DefMathML("Token:?:arcsec",   undef, sub { ['m:arcsec']; }); # OPFUNCTION
DefMathML("Token:?:arccsc",   undef, sub { ['m:arccsc']; }); # OPFUNCTION
DefMathML("Token:?:arccot",   undef, sub { ['m:arccot']; }); # OPFUNCTION
DefMathML("Token:?:arcsinh",  undef, sub { ['m:arcsinh']; }); # OPFUNCTION
DefMathML("Token:?:arccosh",  undef, sub { ['m:arccosh']; }); # OPFUNCTION
DefMathML("Token:?:arctanh",  undef, sub { ['m:arctanh']; }); # OPFUNCTION
DefMathML("Token:?:arcsech",  undef, sub { ['m:arcsech']; }); # OPFUNCTION
DefMathML("Token:?:arccsch",  undef, sub { ['m:arccsch']; }); # OPFUNCTION
DefMathML("Token:?:arccoth",  undef, sub { ['m:arccoth']; }); # OPFUNCTION


# MK (were identical, except for my own typo)

#======================================================================
# Statistics:
#   mean, sdev, variance, median, mode, moment, momentabout

# MK
DefMathML("Token:?:mean",     undef, sub{['m:mean'];}); # OPFUNCTION
DefMathML("Token:?:sdev",     undef, sub{['m:sdev'];}); # OPFUNCTION
DefMathML("Token:?:var",      undef, sub{['m:var'];});	# OPFUNCTION
DefMathML("Token:?:median",   undef, sub{['m:median'];}); # OPFUNCTION
DefMathML("Token:?:mode",     undef, sub{['m:mode'];});	  # OPFUNCTION
DefMathML("Token:?:moment",   undef, sub{['m:moment'];}); # OPFUNCTION

#======================================================================
# Linear Algebra:
#   vector, matrix, matrixrow, determinant, transpose, selector, 
#   vectorproduct, scalarproduct, outerproduct.

# MK
DefMathML("Token:?:vector",        undef, sub{['m:vector'];}); # CONSTRUCTOR
DefMathML("Token:?:matrix",        undef, sub{['m:matrix'];}); # CONSTRUCTOR
DefMathML("Token:?:determinant",   undef, sub{['m:determinant'];}); # OPFUNCTION
DefMathML("Token:?:transpose",     undef, sub{['m:transpose'];});   # OPFUNCTION
DefMathML("Token:?:selector",      undef, sub{['m:selector'];});    # OPFUNCTION
DefMathML("Token:?:vectorproduct", undef, sub{['m:vectorproduct'];}); # OPFUNCTION
DefMathML("Token:?:scalarproduct", undef, sub{['m:scalarproduct'];}); # OPFUNCTION
DefMathML("Token:?:outerproduct",  undef, sub{['m:outerproduct'];}); # OPFUNCTION

#======================================================================
# Semantic Mapping Elements
#   annotation, semantics, annotation-xml
#======================================================================
# Constant and Symbol Elements
#   integers, reals, rationals, naturalnumbers, complexes, primes,
#   exponentiale, imaginaryi, notanumber, true, false, emptyset, pi,
#   eulergamma, infinity

# MK
DefMathML("Token:ID:integers",      undef, sub{['m:integers'];});
DefMathML("Token:ID:reals",         undef, sub{['m:reals'];});
DefMathML("Token:ID:rationals",     undef, sub{['m:rationals'];});
DefMathML("Token:ID:naturalnumbers",undef, sub{['m:naturalnumbers'];});
DefMathML("Token:ID:complexes",     undef, sub{['m:complexes'];});
DefMathML("Token:ID:primes",        undef, sub{['m:primes'];});
DefMathML("Token:ID:exponentiale",  undef, sub{['m:exponentiale'];});
DefMathML("Token:ID:imaginaryi",    undef, sub{['m:imaginaryi'];});
DefMathML("Token:ID:notanumber",    undef, sub{['m:notanumber'];});
DefMathML("Token:ID:true",          undef, sub{['m:true'];});
DefMathML("Token:ID:false",         undef, sub{['m:false'];});
DefMathML("Token:ID:emptyset",      undef, sub{['m:emptyset'];});
DefMathML("Token:ID:pi",            undef, sub{['m:pi'];});
DefMathML("Token:ID:eulergamma",    undef, sub{['m:eulergamma'];});
DefMathML("Token:ID:infinit",       undef, sub{['m:infinit'];});

#======================================================================
# Purely presentational constructs.
# An issue here:
#  Some constructs are pretty purely presentational.  Hopefully, these would
# only appear in XWrap's or in the presentation branch of an XMDual, so we won't
# attempt to convert them to content.  But if we do, should we warn?

DefMathML('Apply:FENCED:?',sub {
  my($op,@elements)=@_;
  pmml_parenthesize(pmml_punctuate($op->getAttribute('separators'),@elements),
		    $op->getAttribute('argopen'), $op->getAttribute('argclose')); });

# Note how annoyingly MML's arrays don't change the style the same
# way TeX does!
DefMathML('Apply:STACKED:?', sub {
  my($op,$over,$under)=@_;
  my $stack = ['m:mtable',{rowspacing=>"0.2ex", columnspacing=>"0.4em"},
	       ['m:mtr',{},['m:mtd',{},pmml($over)]],
	       ['m:mtr',{},['m:mtd',{},pmml($under)]]];
  if($LaTeXML::MathML::STYLE =~/^(text|script)$/){
    ['m:mstyle',{scriptlevel=>'+1'},$stack]; }
  else {
    $stack; }});

# ================================================================================
# cfrac! Ugh!

# Have to deal w/ screwy structure:
# If denom is a sum/diff then last summand can be: cdots, cfrac 
#  or invisibleTimes of cdots and something which could also be a cfrac!
# NOTE: Deal with cfracstyle!!
# OR: Can XMDual build the right stuff?
# AND, the propogation of style is likely wrong...
sub do_cfrac {
  my($numer,$denom)=@_;
  if($denom->localname eq 'XMApp'){ # Denominator is some kind of application
    my ($denomop,@denomargs)=element_nodes($denom);
    if(($denomop->getAttribute('role')||'') eq 'ADDOP'){ # Is it a sum or difference?
      my $last = pop(@denomargs);			# Check last operand in denominator.
      # this is the current contribution to the cfrac (if we match the last term)
      my $curr = ['m:mfrac',{},pmml_smaller($numer),
		  ['m:mrow',{},
		   (@denomargs > 1 ? pmml_infix($denomop,@denomargs) : pmml($denomargs[0])),
		   pmml_smaller($denomop)]];
      if(getTokenMeaning($last) eq 'cdots'){ # Denom ends w/ \cdots
	return ($curr,pmml($last));}		   # bring dots up to toplevel
      elsif($last->localname eq 'XMApp'){	   # Denom ends w/ application --- what kind?
	my($lastop,@lastargs)=element_nodes($last);
	if(getTokenMeaning($lastop) eq 'cfrac'){ # Denom ends w/ cfrac, pull it to toplevel
	  return ($curr,do_cfrac(@lastargs)); }
#	  return ($curr,pmml($last)); }
	elsif((getTokenMeaning($lastop) eq "\x{2062}")  # Denom ends w/ * (invisible)
	      && (scalar(@lastargs)==2) && (getTokenMeaning($lastargs[0]) eq 'cdots')){
	  return ($curr,pmml($lastargs[0]),pmml($lastargs[1])); }}}}
  (['m:mfrac',{},pmml_smaller($numer),pmml_smaller($denom)]); }

DefMathML('Apply:?:cfrac', sub {
  my($op,$numer,$denom)=@_;
  pmml_row(do_cfrac($numer,$denom)); });

#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Specific converters for Presentation, Content, or Parallel.
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

#================================================================================
# Presentation MathML
package LaTeXML::Post::MathML::Presentation;
use strict;
use base qw(LaTeXML::Post::MathML);
use LaTeXML::Util::MathMLLinebreaker;

sub translateNode {
  my($self,$doc,$xmath,$style,$embedding)=@_;
  my @trans = LaTeXML::Post::MathML::pmml_top($xmath,$style);
  my $m = (scalar(@trans)> 1 ? ['m:mrow',{},@trans] : $trans[0]);
  if($style eq 'display'){
    if(my $linelength = $$self{linelength}){
      my $breaker = LaTeXML::Util::MathMLLinebreaker->new();
      if(!$breaker->fitToWidth($m,$linelength,1)){
	my $p = $doc->findnode('ancestor::*[@id]',$xmath);
	my $id = $p && $p->getAttribute('id');
	warn "Couldn't fit math at $id to $linelength"; }}}
  # Wrap unless already embedding within MathML.
  ($embedding =~ /^m:/ ? @trans 
   : ['m:math',{display=>($style eq 'display' ? 'block' : 'inline')},$m]); }

sub getEncodingName { 'MathML-Presentation'; }

#================================================================================
# Content MathML
package LaTeXML::Post::MathML::Content;
use strict;
use base qw(LaTeXML::Post::MathML);

sub translateNode {
  my($self,$doc,$xmath,$style,$embedding)=@_;
  my @trans = LaTeXML::Post::MathML::cmml_top($xmath);
  # Wrap unless already embedding within MathML.
  ($embedding =~ /^m:/ ? @trans 
   : ['m:math',{display=>($style eq 'display' ? 'block' : 'inline')},@trans]); }

sub getEncodingName { 'MathML-Content'; }

#================================================================================
# Parallel MathML
package LaTeXML::Post::MathML::Parallel;
use strict;
use base qw(LaTeXML::Post::MathML);

sub translateNode {
  my($self,$doc,$xmath,$style,$embedding)=@_;
  my($main_proc,@annotation_procs)=@{$$self{math_processors}};
  my @trans = ['m:semantics',{},
	       $main_proc->translateNode($doc,$xmath,$style,'m:semantics'),
	       map( ['m:annotation-xml',{encoding=>$_->getEncodingName},
		     $_->translateNode($doc,$xmath,$style,'m:annotation-xml')],
		    @annotation_procs) ];
  # Wrap unless already embedding within MathML.
  ($embedding =~ /^m:/ ? @trans 
   : ['m:math',{display=>($style eq 'display' ? 'block' : 'inline')},@trans]); }

sub getEncodingName { 'MathML-Parallel'; }

#================================================================================

1;
