# /=====================================================================\ #
# |  LaTeXML::Post::OpenMath                                            | #
# | OpenMath generator for LaTeXML                                      | #
# |=====================================================================| #
# | Part of LaTeXML:                                                    | #
# |  Public domain software, produced as part of work done by the       | #
# |  United States Government & not subject to copyright in the US.     | #
# |---------------------------------------------------------------------| #
# | Bruce Miller <bruce.miller@nist.gov>                        #_#     | #
# | http://dlmf.nist.gov/LaTeXML/                              (o o)    | #
# \=========================================================ooo==U==ooo=/ #

# ================================================================================
# LaTeXML::MathML  Math Formatter for LaTeXML's Parsed Math.
#   Cooperate with the parsed math structure generated by LaTeXML::Math and
# convert into presentation MathML.
# ================================================================================
# TODO
#  * merging of mrows when operator is `close enough' (eg (+ (+ a b) c) => (+ a b c)
#  * get presentation from DUAL
#  * proper parenthesizing (should I record the parens used when parsing?)
# Some clarity to work out:
#  We're trying to convert either parsed or unparsed math (sometimes intertwined).
# How clearly do these have to be separated?
# ================================================================================
package LaTeXML::Post::OpenMath;
use strict;
use LaTeXML::Util::LibXML;
use base qw(LaTeXML::Post);

our $omURI = "http://www.openmath.org/OpenMath";

sub process {
  my($self,$doc)=@_;
  if(my @maths = $self->find_math_nodes($doc)){
    $self->Progress("Converting ".scalar(@maths)." formulae");
    $doc->addNamespace($omURI,'om');
    foreach my $math (@maths){
      $self->processNode($doc,$math); }
    $doc->adjust_latexml_doctype('OpenMath'); } # Add OpenMath if LaTeXML dtd.
  $doc; }

sub find_math_nodes { $_[1]->findnodes('//ltx:Math'); }

# $self->processNode($doc,$mathnode) is the top-level conversion
# It converts the XMath within $mathnode, and adds it to the $mathnode,
sub processNode {
  my($self,$doc,$math)=@_;
  my $mode = $math->getAttribute('mode')||'inline';
  my $xmath = $doc->findnode('ltx:XMath',$math);
  my $style = ($mode eq 'display' ? 'display' : 'text');
  $doc->addNodes($math,$self->translateNode($doc,$xmath,$style,'ltx:Math')); }

sub translateNode {
  my($self,$doc,$xmath,$style,$embedding)=@_;
  my @trans = Expr($xmath);
  # Wrap unless already embedding within MathML.
  ($embedding =~ /^om:/ ? @trans : ['om:OMOBJ',{},@trans]); }

sub getEncodingName { 'OpenMath'; }
# ================================================================================

sub getTokenMeaning {
  my($node)=@_;
  my $m = $node->getAttribute('meaning') || $node->getAttribute('name')
    || $node->textContent;
  (defined $m ? $m : '?'); }

sub getTokenName {
  my($node)=@_;
  my $m = $node->getAttribute('name') || $node->textContent;
  (defined $m ? $m : '?'); }

# ================================================================================
our $OMTable={};

sub DefOpenMath {
  my($key,$sub) =@_;
  $$OMTable{$key} = $sub; }

sub Expr {
  my($node)=@_;
  return OMError("Missing Subexpression") unless $node;
  my $tag = $node->nodeName;
  if($tag eq 'XMath'){
    my($item,@rest)=  element_nodes($node);
    print STDERR "Warning! got extra nodes for content!\n" if @rest;
    Expr($item); }
  elsif($tag eq 'XMDual'){
    my($content,$presentation) = element_nodes($node);
    Expr($content); }
  elsif($tag eq 'XMWrap'){
    # Note... Error?
##    Row(grep($_,map(Expr($_),element_nodes($node)))); 
    (); }
  elsif($tag eq 'XMApp'){
    my($op,@args) = element_nodes($node);
    return OMError("Missing Operator") unless $op;
    my $name =  getTokenMeaning($op);
    my $pos  =  $op->getAttribute('role') || '?';

    my $sub = $$OMTable{"Apply:$pos:$name"} || $$OMTable{"Apply:?:$name"} 
      || $$OMTable{"Apply:$pos:?"} || $$OMTable{"Apply:?:?"};
    &$sub($op,@args); }
  elsif($tag eq 'XMTok'){
    my $name =  getTokenMeaning($node);
    my $pos  =  $node->getAttribute('role') || '?';
    my $sub = $$OMTable{"Token:$pos:$name"} || $$OMTable{"Token:?:$name"} 
      || $$OMTable{"Token:$pos:?"} || $$OMTable{"Token:?:?"};
    &$sub($node); }
  elsif($tag eq 'XMHint'){
    (); }
  else {
    ['om:OMSTR',{},$node->textContent]; }}

# ================================================================================
# Helpers
sub OMError {
  my($msg)=@_;
  ['om:OME',{},
   ['om:OMS',{name=>'unexpected', cd=>'moreerrors'}],
   ['om:OMS',{},$msg]]; }
# ================================================================================
# Tokens

# Note: In general, there needs to be a lot more support/analysis.
# Here, we simply assume that the token is a variable if there's no CD!!!
DefOpenMath('Token:?:?',    sub { 
  my($token)=@_;
  my $name = getTokenMeaning($token);
  my $cd = $token->getAttribute('omcd');
  if($cd){
    ['om:OMS',{name=>$name, cd=>$cd}]; }
  else {
    ['om:OMV',{name=>$name}]; }});

# NOTE: Presence of '.' distinguishes float from int !?!?
DefOpenMath('Token:NUMBER:?',sub {
  my($node)=@_;
  my $value = getTokenMeaning($node); # name attribute (may) holds actual value.
  if($value =~ /\./){
    ['om:OMF',{dec=>$value}]; }
  else {
    ['om:OMI',{},$value]; }});

DefOpenMath("Token:?:\x{2062}", sub {
  ['om:OMS',{name=>'times', cd=>'arith1'}]; });

# ================================================================================
# Applications.

# Generic

DefOpenMath('Apply:?:?', sub {
  my($op,@args)=@_;
  ['om:OMA',{},map(Expr($_),$op,@args)]; });

# NOTE: No support for OMATTR here...

# NOTE: Sketch of what OMBIND support might look like.
# Currently, no such construct is created in LaTeXML...
DefOpenMath('Apply:LambdaBinding:?', sub {
  my($op,$expr,@vars)=@_;
  ['om:OMBIND',{},
   ['om:OMS',{name=>"lambda", cd=>'fns1'},
    ['om:OMBVAR',{},map(Expr($_),@vars)], # Presumably, these yield OMV
    Expr($expr)]]; });

# ================================================================================
1;
