﻿// card.h
#ifndef CARD_H_INCLUDED
#define CARD_H_INCLUDED

#include <string>
#include <vector>
#include "stream.h"

#define CARD_MARK_STRING_EMOJI      0   // 絵文字
#define CARD_MARK_STRING_KANA       1   // カタカナ
#define CARD_MARK_STRING_ALPHABET   2   // アルファベット
#define CARD_MARK_STRING            CARD_MARK_STRING_EMOJI  // カードマークの文字列表現の方法

// カードの数字の型
using CardNumber = signed char;

// カードのマークの型
enum class CardMark : signed char {
    spade,
    club,
    diamond,
    heart,
};

// カード型
struct Card {
    CardNumber number;  // 数字
    CardMark mark;      // マーク
};

// カードの集まりを表す型
using cards_type = std::vector<Card>;

constexpr auto card_mark_num = 4;                                    // マークの総数
constexpr auto each_mark_card_num = 13;                              // 各マークのカードの枚数
constexpr auto trump_card_num = each_mark_card_num * card_mark_num;  // トランプに含まれるカードの枚数


// トランプを初期化する
// cards: カードの集まり
void init_trump(cards_type& cards);

// カードをシャッフルする
// cards: カードの集まり
void shuffle_cards(cards_type& cards);

// カードを整列する
// cards: カードの集まり
void sort_cards(cards_type& cards);

// カードの整列に使う比較関数
// a: 対象のカード
// b: 対象のカード
// 戻り値: a と b が適切な順番で並んでいたら true、並んでいなければ false
bool cards_sort_compare(const Card& a, const Card& b);

// カードの数字の文字列表現を返す
// たとえば、3 は "3"、1 は "A"、13 は "K" と表現される。
//
// number: カードの数字
// 戻り値: カードの数字を文字列で表現したもの
utf8_string get_card_number_string(CardNumber number);

// カードのマークの文字列表現を返す
// 具体的な表現方法は、CARD_MARK_STRING マクロの定義によって異なる。
// 
// card_mark: カードのマーク
// 戻り値: カードのマークを文字列で表現したもの
utf8_string get_mark_string(CardMark card_mark);

#endif
