﻿// main.cpp
#include <iostream>
#include <sstream>
#ifdef _WIN32
#define NOMINMAX
#include <Windows.h>
#endif
#include "card.h"
#include "poker_hand.h"
#include "stream.h"

constexpr CardNumber nothing_card_number = -1;  // カードがないことを表すダミー値

// 手札を配る
static void hand_out_cards(cards_type& deck, cards_type& hand_cards);

// カードを補充する
static void reload_cards(cards_type& deck, cards_type& hand_cards);

// カードの情報を出力する
static void print_cards(const cards_type& cards);

// 捨てるカードを選ぶときの出力
static void print_choose_discard(const cards_type& cards, const std::vector<bool>& is_discard);

// カードの文字列表現を返す
static utf8_string get_card_string(const Card& card);


int main()
{
#ifdef _WIN32
    SetConsoleOutputCP(CP_UTF8);  // コマンドプロンプトでの表示に UTF-8 を使う
#endif

    while (true) {

        // 山札を準備
        cards_type deck(trump_card_num);
        init_trump(deck);
        shuffle_cards(deck);

        // 最初の手札を配る
        cards_type player_cards {};
        hand_out_cards(deck, player_cards);

        // ユーザーにカードを提示
        std::cout << u8"あなたに配られたカードです。\n";
        print_cards(player_cards);

        // 捨てるカードを選ばせる
        std::vector<bool> is_discard(hand_card_num);
        while (true) {
            print_choose_discard(player_cards, is_discard);

            std::string input_string {};
            std::getline(std::cin, input_string);
            if (!std::cin) {
                std::cout << u8"入力が正しくありません。\n";
            }
            else if (input_string.empty()) {
                break;
            }
            else {
                int index {input_string[0] - '0'};  // 数字から整数に変換
                if (0 <= index && index < hand_card_num) {
                    is_discard[index] = !is_discard[index];
                }
                else {
                    std::cout << u8"入力が正しくありません。\n";
                }
            }
        }

        // 選ばれたカードを捨てる
        for (std::vector<bool>::size_type i = 0; i < is_discard.size(); ++i) {
            if (is_discard.at(i)) {
                auto& card = player_cards.at(i);
                std::cout << get_card_string(card) << u8" を捨てました。\n";

                // 何もないことをあらわすダミー値を入れる
                card.number = nothing_card_number;
            }
        }
        std::cout << u8"\n";

        // 新しいカードを補充する
        reload_cards(deck, player_cards);
        sort_cards(player_cards);
        std::cout << u8"\n";
        print_cards(player_cards);

        // 役判定
        auto poker_hand = judge_poker_hand(player_cards);
        if (poker_hand == PokerHand::no_pair) {
            std::cout << u8"残念でした。\n";
        }
        else {
            std::cout << get_poker_hand_string(poker_hand) << u8"ができました。\n";
        }

        // ゲームの続行確認
        std::cout << u8"\n"
                  << u8"ゲームを続けますか？\n"
                  << u8"はい: Y  いいえ: N\n";
        std::string input_string {};
        std::getline(std::cin, input_string);
        if (input_string == "Y" || input_string == "y") {
            continue;
        }
        else {
            break;
        }
    }
}


// 手札を配る
static void hand_out_cards(cards_type& deck, cards_type& hand_cards)
{
    while (hand_cards.size() < hand_card_num) {
        hand_cards.push_back(deck.back());
        deck.pop_back();
    }
    sort_cards(hand_cards);
}

// カードを補充する
static void reload_cards(cards_type& deck, cards_type& hand_cards)
{
    // 何もないことをあらわすダミー値が入っているところに、新しいカードを補充する
    for (cards_type::size_type i = 0; i < hand_cards.size(); ++i) {
        if (hand_cards.at(i).number == nothing_card_number) {
            auto& card = deck.back();

            std::cout << get_card_string(card) << u8" が配られました。\n";
            hand_cards.at(i) = card;
            deck.pop_back();
        }
    }
}

// カードの情報を出力する
static void print_cards(const cards_type& cards)
{
    for (cards_type::size_type i = 0; i < cards.size(); ++i) {
        auto& card {cards.at(i)};

        if (card.number == nothing_card_number) {
            std::cout << i << u8": \n";
        }
        else {
            std::cout << i << u8": " << get_card_string(card) << u8"\n";
        }
    }
}

// 捨てるカードを選ぶときの出力
static void print_choose_discard(const cards_type& cards, const std::vector<bool>& is_discard)
{
    std::cout << u8"\n"
              << u8"左端の番号を入力して、カードを選んでください。\n"
              << u8"何も入力せず Enterキーを押すと決定します。\n";

    for (cards_type::size_type i = 0; i < cards.size(); ++i) {
        std::cout << i << u8": " << get_card_string(cards.at(i));

        // 捨てるカードに含まれているか
        if (is_discard[i]) {
            std::cout << u8" <-- 捨てる";
        }

        std::cout << u8"\n";
    }
}

// カードの文字列表現を返す
static utf8_string get_card_string(const Card& card)
{
#if CARD_MARK_STRING == CARD_MARK_STRING_KANA
    return get_mark_string(card.mark) + u8"の" + get_card_number_string(card.number);
#else
    return get_mark_string(card.mark) + u8" " + get_card_number_string(card.number);
#endif
}
