function F=makeRFSfilters
% Returns the RFS filter bank of size 49x49x38 in F. The MR8, MR4 and
% MRS4 sets are all derived from this filter bank. To convolve an
% image I with the filter bank you can either use the matlab function
% conv2, i.e. responses(:,:,i)=conv2(I,F(:,:,i),'valid'), or use the
% Fourier transform.

% from Varma web site:
% To achieve rotational invariance, derive Maximum Response 8 (MR8) filter
% bank from RFS by recording only the maximum filter response across all
% orientations for the two anisotropic filters.  This reduces number of
% responses from 38 (6 orientations at 3 sclaes for 2 oriented filters,
% plus 2 isotropic) to 8 (3 sclaes for 2 filters, plus 2 isotropic).  Thus
% the MR8 filter bank consists of 38 filters but only 8 filter responses.
%
% http://www.robots.ox.ac.uk/~vgg/research/texclass/filters.html

  SUP=49;                 % Support of the largest filter (must be odd)
  SCALEX=[1,2,4];         % Sigma_{x} for the oriented filters
  NORIENT=6;              % Number of orientations

  NROTINV=2;
  NBAR=length(SCALEX)*NORIENT;
  NEDGE=length(SCALEX)*NORIENT;
  NF=NBAR+NEDGE+NROTINV;
  F=zeros(SUP,SUP,NF);
  hsup=(SUP-1)/2;
  [x,y]=meshgrid([-hsup:hsup],[hsup:-1:-hsup]);
  orgpts=[x(:) y(:)]';

  
  % first 36 filters are edges and bars at different orientations and
  % scales, orientations 1:6 at scale 1, orientations 1:6 at scale 2, ...
  count=1;
  for scale=1:length(SCALEX),
    for orient=0:NORIENT-1,
      angle=pi*orient/NORIENT;  % Not 2pi as filters have symmetry
      c=cos(angle);s=sin(angle);
      rotpts=[c -s;s c]*orgpts;
      F(:,:,count)=makefilter(SCALEX(scale),0,1,rotpts,SUP);
      F(:,:,count+NEDGE)=makefilter(SCALEX(scale),0,2,rotpts,SUP);
      count=count+1;
    end;
  end;   % this makes 36 filters: 3 scales * 6 orientations = 18 * 2 (edge/bar) = 36

  F(:,:,NBAR+NEDGE+1)=normalise(fspecial('gaussian',SUP,10));
  F(:,:,NBAR+NEDGE+2)=normalise(fspecial('log',SUP,10));
  % last two filters (37 and 38) are isotropic / rotationally symmetric:
  % Gaussian and LoG.
return

function f=makefilter(scale,phasex,phasey,pts,sup)
  gx=gauss1d(3*scale,0,pts(1,:),phasex);
  gy=gauss1d(scale,0,pts(2,:),phasey);
  f=normalise(reshape(gx.*gy,sup,sup));
return

function g=gauss1d(sigma,mean,x,ord)
% Function to compute gaussian derivatives of order 0 <= ord < 3
% evaluated at x.

  x=x-mean;num=x.*x;
  variance=sigma^2;
  denom=2*variance; 
  g=exp(-num/denom)/sqrt(pi*denom);
  switch ord,
    case 1, g=-g.*(x/variance);
    case 2, g=g.*((num-variance)/(variance^2));
  end;
return

function f=normalise(f), f=f-mean(f(:)); f=f/sum(abs(f(:))); return